ad_library {
    
    @author Joel Aufrecht (joel@aufrecht.org)
    @creation-date 2004-02-08
    @cvs-id $Id: vocabulary-train-procs.tcl,v 1.1.1.1 2004/02/23 23:00:08 joela Exp $
    
}

#---------------------------------------------------------------------
# vocab::train
#---------------------------------------------------------------------
namespace eval vocab::train {}

#---------------------------------------------------------------------
ad_proc -public vocab::train::test_type_list {
} {
    Get a list of all test types.

    @author Joel Aufrecht
    @creation-date 2004-02-18
    @return list of pairs of label and test_type
} {
    lappend list [list sentence sentence]
    lappend list [list sentence_group sentence_group]
    lappend list [list sentence_from_hint sentence_from_hint]
    lappend list [list ortho_from_ortho  ortho_from_ortho]
    return $list
}

#---------------------------------------------------------------------
# vocab::train::test
#---------------------------------------------------------------------
namespace eval vocab::train::test {}

ad_proc -public vocab::train::test::new {
    -package_id:required
    -test_type:required
    {-test_subject ""}
    {-question ""}
    {-answer ""}
    {-locale_a ""}
    {-locale_b ""}
} {
    Create a new test.  Ste the question and answer values.  TODO: if test_subject is a list, create multiple records in vocab_test_subject.

    @author Joel Aufrecht
    @creation-date 2004-02-16
    @return ID of new test, or 0 if there was a problem.
    
} {
    # make sure we have all of the data necessary, and get it if we can
    switch $test_type {
        sentence {
            # this isn't implemented yet
            return 0
            vocab::sentence::get -id $test_subject -array sentence_array
            set row(question) "not implemented yet" ;#TODO: get the locale_b translation of a sentence as question,
            set row(answer) $sentence_array(text)
        }
        sentence_from_hint {
            if { ![exists_and_not_null question] } {
                return 0
            }
            vocab::sentence::get -id $test_subject -array sentence_array
            if { ![exists_and_not_null answer] } {
                set answer $sentence_array(text)
            }
        }
        ortho_from_ortho { 
            if { ![exists_and_not_null answer] || ![exists_and_not_null question] } {
                vocab::word::get \
                    -id $test_subject \
                    -locale_b $locale_b \
                    -array word
                if { ![exists_and_not_null question] } {
                    if { [exists_and_not_null word(local_phonetic)] } {
                        set phonetic "/<em>$word(local_phonetic)</em>/"
                    } elseif { [exists_and_not_null word(ipa_phonetic)] } {
                        set phonetic "/<em>$word(ipa_phonetic)</em>/"
                    } else {
                        set phonetic ""
                    }
                    set label [ad_locale_get_label $locale_b]
                    set question "What does <em>$word(word)</em> $phonetic mean in $label?"
                }
                if { ![exists_and_not_null answer] } {
                    set answer [list $word(definitions)]
                }
            }
        }
        default  {
            return 0
        }
    }
    set test_id [db_string get_next_testid "
                 select nextval('vocab_test_id_seq')
                 "]

    db_dml insert_test "insert into vocab_test
    (id, package_id, test_type, test_subject, question, answer, locale_a, locale_b) 
    values (:test_id, :package_id, :test_type, :test_subject, :question, :answer, :locale_a, :locale_b)"
    return $test_id
}


ad_proc -public vocab::train::test::get {
    -test_id:required
    -array:required
    -target_locale:required
} {
    Return the question and answer for a test, depending on test type.
    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return The test id.

    @param test_id The test's unique ID
    @param array The name of an array to return question, answer.
    @param target_locale The locale_b of the answer.
} {
    upvar 1 $array row

    db_1row get_test_info "
                   select test_type,
                          test_subject,
                          question,
                          answer,
                          locale_a
                     from vocab_test
                    where id = :test_id
                      and locale_b = :target_locale
    " -column_array row
    return $test_id
}

ad_proc -public vocab::train::test::delete {
    -test_id:required
} {
    Return the question and answer for a test, depending on test type.
    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return 0 for success; 1 for failure.

    @param test_id The test's unique ID
} {

    # TODO: not checking permission here yet; instead just assuming that 
    # this can only get called by an admin
    db_transaction {
        db_dml delete_test_results "
               delete
                 from vocab_test_result
                where test_id = :test_id"

        db_dml delete_test "
               delete
                 from vocab_test
                where id = :test_id"
    }
}

#---------------------------------------------------------------------
# vocab::train::result
#---------------------------------------------------------------------
namespace eval vocab::train::result {}

ad_proc -public vocab::train::result::submit {
    -test_id:required
    -user_id:required
    {-date  ""}
    -answer1:required
    -array:required
    -target_locale
} {
    Accept a user's answer to a test question, grade it, store and return the results.

    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return id of new test result
    
    @param test_id The intended test id
    @param answer1 The user's answer to the question.
    @param date The date of the answer.  Defaults to now.
    @param array The name of an array to return quesion, user's answer, correct answer, and score.
} {
    if {![exists_and_not_null date]} {
        set date [clock_to_ansi [clock seconds]]
    }
    upvar 1 $array row

    vocab::train::test::get \
        -test_id $test_id \
        -array answer_key \
        -target_locale $target_locale

    # make sure we have everything we need
    if {  [exists_and_not_null answer_key(answer)] && 
          ![string equal $answer_key(answer) ""] &&
          [exists_and_not_null answer_key(test_type)]  } {

        switch $answer_key(test_type) {
            sentence_from_hint {
                set correct_answer $answer_key(answer)
                if { [string equal $correct_answer $answer1] } {
                    set score 1
                } else {
                    set score 0
                }
            }
            ortho_from_ortho { 
                # see if the answer is in the list
                set answer [join $answer_key(answer)]
                # lsearch returns -1 for no match, 0 for match on 1st item
                if { [lsearch $answer $answer1] > -1 } {
                    set score 1
                } else {
                    set score 0
                }
                if { [llength $answer_key(answer)] > 0 } { 
                    set correct_answer "[join $answer ", "]"
                } else {
                    set correct_answer $answer_key(answer)
                }
            }
            default  {
                format "question missing" 
            }
        }
        db_dml set_answer "
            insert into vocab_test_result 
                   (test_id, user_id, date, answer1, result1)
            values (:test_id, :user_id, :date, :answer1, :score)
         "
        array set row [list question $answer_key(question)]
        array set row [list user_answer $answer1]
	array set row [list correct_answer $correct_answer]
        array set row [list score $score]
        array set row [list date $date]
        array set row [list test_subject $answer_key(test_subject)]
    } else {
        array set row [list question $answer_key(question)]
        array set row [list user_answer $answer1]
        array set row [list correct_answer "No correct answer defined."]
    }
    return 0
}
