# packages/vocabulary/tcl/vocabulary-procs.tcl

ad_library {
    
    @author Joel Aufrecht (joel@aufrecht.org)
    @creation-date 2004-02-02
    @cvs-id $Id: vocabulary-procs.tcl,v 1.1.1.1 2004/02/23 23:00:08 joela Exp $
    
}

#####################################################################
# vocab
######################################################################

namespace eval vocab {}

#####################################################################
# vocab::word
######################################################################

namespace eval vocab::word {}

#---------------------------------------------------------------------
ad_proc -public vocab::word::new {
    -package_id:required
    -word:required
    -locale:required
    -user_id:required
    {-ipa_phonetic ""}
    {-local_phonetic ""}
    {-def_locale ""}
    {-def_list ""}
} {
    Insert a word into the system.  Insert definitions if provided.  Create tests.  If locale is Chinese, pinyin with numbers is changed to unicode.

    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return The object id of the new word.

    @param def_list A comma-separated list of definitions that apply to the word.
    @param def_locale The locale of all of the definitions in def_list.  If def_locale is null, def_list is ignored.
} {

    if { [string equal [string tolower $locale] "ch_zh"] && [exists_and_not_null local_phonetic] } {
        set local_phonetic [vocab::locale::pinyin_num_to_unicode -string $local_phonetic]
    }
    set id [db_string word_insert "
        select vocab_word__new(:package_id, :word, :locale, :ipa_phonetic, :local_phonetic, :user_id,null) 
    "]

    if { [exists_and_not_null def_list] } {
        if { ![exists_and_not_null def_locale]} {
            # defs but no def_locale?  Give up.
            return $id
        }

        # TODO: either here or in def::new, should check for
        # existing words and try to match
        # and try to link to existing words instead of creating def text

        foreach def [split $def_list ,] {
            vocab::word::def::new \
                -word_id $id \
                -locale $def_locale \
                -definition [string trim $def]
        }

        vocab::train::test::new \
            -package_id $package_id \
            -test_type "ortho_from_ortho" \
            -test_subject $id \
            -locale_a $locale \
            -locale_b $def_locale
    }
    return $id
}

#---------------------------------------------------------------------
ad_proc -public vocab::word::edit {
    -id:required
    -word:required
    -locale:required
    -ipa_phonetic
    -local_phonetic
} {
    Edit a word into the system.

    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return The object id of the new word.

} {
    db_dml word_edit {
        update vocab_word
           set word   = :word,
               locale = :locale,
               ipa_phonetic = :ipa_phonetic,
               local_phonetic = :local_phonetic
         where id = :id
    }
    return $id
}

#---------------------------------------------------------------------
ad_proc -public vocab::word::get {
    -id:required
    -array:required
    {-locale_b ""}
} {
    Return word, ipa_phonetic, local_phonetic, and a list of definitions in the specified locale.

    @author Joel Aufrecht
    @creation-date 2004-02-07
    @return id for success, -1 for failure.

    @param array The name of an array to return
} {
    upvar 1 $array row
    set got_word [db_0or1row question_select "
                select vw.word,
                       vw.locale,
                       vw.ipa_phonetic,
                       vw.local_phonetic
                  from vocab_word vw
                 where vw.id = :id
    "  -column_array row]

    if { !$got_word} {
        return -1
    }

    if { [exists_and_not_null locale_b] } {
        set def_list [db_list select_def_list "
                        select definition 
                          from vocab_definition 
                         where word_id = :id
                           and locale = :locale_b
                      "]
    }

    if { [exists_and_not_null def_list] } {
        array set row [list definitions $def_list]
    } else {
        array set row [list definitions ""]
    }
    return $id
}

ad_proc -public vocab::word::delete {
    -id:required
} { 
    This proc deletes a word.
    @return 0 for success
} {
    # TODO: improve security
    auth::require_login
    db_transaction {
        db_dml word_defs_delete "
            delete 
              from vocab_definition 
             where word_id = :id"
        db_dml word_phonetic_example_delete "
            delete
              from vocab_phonetic_example
             where word_id = :id"
        db_dml word_word_list_delete "
            delete
              from vocab_word_list
             where word_id = :id"

        db_exec_plsql word_delete "
	select vocab_word__delete(:id)
    "
    }
    return 0
}


#####################################################################
# vocab::word::def
######################################################################

namespace eval vocab::word::def {}

#---------------------------------------------------------------------
ad_proc -public vocab::word::def::new {
    -word_id:required
    -locale:required
    -definition:required
} {
    Add a definition to a word.

    @author Joel Aufrecht
    @creation-date 2004-02-10
    @return 0 for success
    
} {
    db_dml definition_insert {
        insert into vocab_definition
        (word_id, locale, definition)
        values (:word_id, :locale, :definition)
    }
}

#---------------------------------------------------------------------
ad_proc -public vocab::word::def::get {
    -id:required
    -array:required
} {
    Get a definition.

    @author Joel Aufrecht
    @creation-date 2004-02-10
    @return 0 for success

    @param array The name of an array to return    
} {
    upvar 1 $array row   
    db_1row definition_select {
        select vw.word,
               vd.word_id,
               vd.sort_order,
               vd.locale,
               vd.definition
          from vocab_definition vd,
               vocab_word vw
         where vd.id = :id
           and vw.id = vd.word_id
    } -column_array row

    return 0
}

#---------------------------------------------------------------------
ad_proc -public vocab::word::def::get_by_word {
    -word:required
    -locale:required
} {
    Get a definition.

    @author Joel Aufrecht
    @creation-date 2004-02-10
    @return A list of definitions for the given word in the given locale.

} {
    return [db_list get_defs "
            select vd.definition
              from vocab_definition vd,
	           vocab_word vw
             where vw.word = :word
               and vw.id = vd.word_id
               and vd.locale = :locale"]
}


#---------------------------------------------------------------------
ad_proc -public vocab::word::def::delete {
    -id:required
} { 
    This proc deletes a definition.
    @return 0 for success
} {
    auth::require_login
    db_dml def_delete "
        delete
          from vocab_definition
         where id = :id"
    }
    return 0
}


