#
#  Copyright (C) 2001, 2002 MIT
#
#  This file is part of dotLRN.
#
#  dotLRN is free software; you can redistribute it and/or modify it under the
#  terms of the GNU General Public License as published by the Free Software
#  Foundation; either version 2 of the License, or (at your option) any later
#  version.
#
#  dotLRN is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#

ad_library {

    portal procs

    @author <a href="mailto:arjun@openforce.net">arjun@openforce.net</a>
    @author <a href="mailto:yon@openforce.net">yon@openforce.net</a>
    @creation-date 2001-09-01
    @version $Id: portal-procs.tcl,v 1.21 2004/03/17 02:15:53 donb Exp $

}

namespace eval portal {

    ad_proc -public mount_point {} {
        caches the mount point
    } {
        return [lindex [site_node::get_url_from_object_id -object_id [ad_conn package_id]] 0]
    }

    ad_proc -public automount_point {} {
        packages such as dotlrn can automount the portal here
    } {
        return portal
    }

    ad_proc -public new {
        {-owner_id:required}
        {-name Untitled}
        {-template_id ""}
        {-context_id ""}
        {-page_list {{{Page 1} 2_column}}}
        {-owner_privileges {read write admin}}
        {-initialize_elements:boolean}
        {-theme_name ""}
    } {
        Create a new portal for the passed in owner_id. create pages passed in
        the page_list.

        @owner_id The object (usually a party) which owns the new portal
        @name The name of the portal
        @template_id If set, copy this portal's elements into the new portal
        @context_id  The new portal object's context_id (defaults to package_id)
        @page_list The list of pages to create within the portal
        @owner_privileges A list of privileges to grant the party on the new portal.
          Normally the default list (read, write, admin) will be correct if the party
          is a user but if the party is a group (say, members of a subsite) then it
          would make sense to only grant read to the party directly.
        @initialize_elements_p If true and template_id is set, initialize elements after
          copying.
        @theme_name The theme to assign to this portal (defaults to the package's
          default_theme_name)

    } {

        if { [string equal $context_id ""] } {
            set context_id [ad_conn package_id]
        }

        if { [string equal $theme_name ""] } {
            set theme_name [parameter::get -parameter DefaultThemeName]
        }

        db_transaction {

            set var_list [subst {
                {name $name}
                {template_id $template_id}
                {context_id $context_id}
                {owner_id $owner_id}
                {theme_id [portal::theme::get_id -name $theme_name]}
                {package_id [ad_conn package_id]}}]

            set portal_id [package_instantiate_object -var_list $var_list portal]

            foreach privilege $owner_privileges {
                permission::grant -party_id $owner_id -object_id $portal_id -privilege $privilege
            }

            if {[empty_string_p $template_id]} {
                foreach page $page_list {
                    portal::page::new \
                        -portal_id $portal_id \
                        -name [lindex $page 0] \
                        -layout_name [lindex $page 1]
                }
            }

            if { $initialize_elements_p && ![string equal $template_id ""] } {
                foreach row [db_list_of_lists get_elements {}] {
                    portal::datasource::call \
                        -name [lindex $row 0] \
                        -op Initialize \
                        -list_args [list [lindex $row 1]]
                }
            }

        }

        return $portal_id
    }

    ad_proc -public delete {
        {-portal_id:required}
    } {
        delete a portal
    } {
        db_dml delete_permissions {}
        db_exec_plsql delete_portal {}
    }

    ad_proc -public get {
        {-portal_id:required}
    } {
        get portal info in "array get" format (cached)
    } {
        return [util_memoize "portal::get_not_cached -portal_id $portal_id"]
    }

    ad_proc -private get_not_cached {
        {-portal_id:required}
    } {
        get portal info in "array get" format (not cached)
    } {
        db_1row select_portal_info {} -column_array portal
        return [array get portal]
    }

    ad_proc -public get_user_portal_id {
        -package_id
    } {
        Get the portal_id for a user. I (DRB) am not going to bother caching this
        for the time being ... we'll nsv this later.

        @param package_id The package_id of the portal instance (defaults to ad_conn package_id)
        @return The portal_id for the user or the master template portal_id if personal
            portals are not enabled.
    } {

        if { ![info exists package_id] } {
            set package_id [ad_conn package_id]
        }

        if { [parameter::get -package_id $package_id -parameter CreatePrivatePortals] } {
            set owner_id [ad_conn user_id]
        } else {
            set owner_id 0
        }

        if {![db_0or1row select_portal_id {}]} {

            if { ![portal::initialized -package_id $package_id] } {
                if { [permission::permission_p -object_id $package_id -privilege admin] } {
                    ad_returnredirect admin
                    ad_script_abort
                } else {
                    ad_return_exception_template -params {{custom_message "The portal package hasn't been configured yet."}} /packages/acs-subsite/www/shared/report-error
                }
            }

            # At this point we know we're supposed to create a personal portal

            db_1row select_user_name {}

            set master_template_id [portal::get_master_template_id -package_id $package_id]

            set portal_id [portal::new \
                -owner_id :owner_id \
                -name "Portal for $user_name" \
                -template_id $master_template_id \
                -context_id [ad_conn package_id] \
                -initialize_elements]

        }
        return $portal_id
    }

    ad_proc -public set_values {
        {-portal:required}
    } {
        set the fields of a portal
    } {
        array set portal_array $portal
        template::util::array_to_vars portal_array

        db_dml update_portal {}
        util_memoize_flush "portal::get_not_cached -portal_id $portal_id"
    }

    ad_proc -public get_name {
        {-portal_id:required}
    } {
        array set portal [get -portal_id $portal_id]
        return $portal(name)
    }

    ad_proc -public set_name {
        {-portal_id:required}
        {-name:required}
    } {
        array set portal [get -portal_id $portal_id]
        set portal(name) $name

        set_values -portal [array get portal]
    }

    ad_proc -public get_theme_id {
        {-portal_id:required}
    } {
        array set portal [get -portal_id $portal_id]
        return $portal(theme_id)
    }

    ad_proc -public set_theme_id {
        {-portal_id:required}
        {-theme_id:required}
    } {
        array set portal [get -portal_id $portal_id]
        set portal(theme_id) $theme_id

        set_values -portal [array get portal]
    }

    ad_proc -private get_template_id {
        {-portal_id:required}
    } {
        array set portal [get -portal_id $portal_id]
        return $portal(template_id)
    }

    ad_proc -public get_master_template_id {
        -package_id
    } {
        Get the master template id, i.e. the portal used to create all other portals.

        @param package_id The package_id of the portal instance (defaults to ad_conn package_id)
        @return The portal_id of the master portal template.

    } {

        if { ![info exists package_id] } {
            set package_id [ad_conn package_id]
        }

        if { [db_0or1row get_master_template_id {}] } {
            return $master_template_id
        } else {
            return ""
        }
    }

    ad_proc -public get_admin_portal_id {
        -package_id
    } {
        Get the admin portal id.

        @param package_id The package_id of the portal instance (defaults to ad_conn package_id)
        @return The portal_id of the admin portal.

    } {

        if { ![info exists package_id] } {
            set package_id [ad_conn package_id]
        }

        if { [db_0or1row get_admin_portal_id {}] } {
            return $admin_portal_id
        } else {
            return ""
        }
    }

    ad_proc -public get_page_count {
        {-portal_id:required}
    } {
        return [db_string select_page_count {}]
    }

    ad_proc -public get_page_list {
        {-portal_id:required}
    } {
        @return list of page_ids associated with the given portal (in sort_key order).
    } {
        return [db_list select_page_ids {}]
    }

    ad_proc -public get_render_data {
        {-portal_id:required}
        {-page_num ""}
        {-render_style individual}
    } {
        returns metadata needed to render a portal

        @portal_id The id of the portal you're interested in.
        @page_num The page within the portal (defaults to page 0).
        @render_style Either "individual" or "all-in-one".

        @return The portal metadata in array get format.
    } {
        if { [string equal $page_num ""] } {
            set page_num 0
        }
        # get the portal and layout
        db_1row portal_select {} -column_array portal
        set page_id $portal(page_id)
        set portal(page_src) "/packages/portal/lib/render-styles/${render_style}/render-page"
        return [array get portal]
    }

    ad_proc -private layout_elements {
        element_list
        {var_stub "element_ids"}
    } {
        Split a list up into a bunch of variables for inserting into a
        layout template. This seems pretty kludgy (probably because it is),
        but a template::multirow isn't really well suited to data of this
        shape. It'll setup a set of variables, $var_stub_1 - $var_stub_8
        and $var_stub_i1- $var_stub_i8, each contining the portal_ids that
        belong in that region. - Ian Baker

        @param element_id_list An [array get]'d array, keys are regions, \
                values are lists of element_ids.
        @param var_stub A name upon which to graft the bits that will be \
                passed to the template.
    } {
       array set elements $element_list

        foreach idx [list 1 2 3 4 5 6] {
            upvar "${var_stub}_$idx" group
            if { [info exists elements($idx) ] } {
                set group $elements($idx)
            } else {
                set group {}
            }
        }
    }


    ad_proc -public configure_dispatch {
        {-template_p f}
        {-portal_id:required}
        {-form:required}
    } {
        Dispatches the configuration operation.
        We get the target region number from the op.

        @param portal_id the portal to edit
        @param formdata an ns_set with all the formdata
    } {

        permission::require_permission -object_id $portal_id -privilege write

        set op [ns_set get $form op]

        switch $op {
            "Revert" {
                db_transaction {
                    set template_id [get_template_id -portal_id $portal_id]

                    # revert theme
                    set_theme_id -portal_id $portal_id -theme_id [get_theme_id -portal_id $template_id]

                    # revert pages
                    # first equalize number of pages in the target
                    set template_page_count [get_page_count -portal_id $template_id]
                    set target_page_count [get_page_count -portal_id $portal_id]
                    set difference [expr $template_page_count - $target_page_count]

                    if {$difference > 0} {
                        # less pages in target
                        for {set x 0} {$x < $difference} {incr x} {

                            set name "portal revert dummy page $x"
                            portal::page::new \
                                -portal_id $portal_id \
                                -name $name \
                        }
                    } elseif {$difference < 0} {
                        # more pages in target, delete them from the end,
                        # putting any elements on them on the first page,
                        # we put them in the right place later
                        for {set x 0} {$x < [expr abs($difference)]} {incr x} {

                            set max_page_id [db_string revert_max_page_id_select {}]
                            set page_id [db_string revert_min_page_id_select {}]
                            set region 1

                            db_foreach revert_move_elements_for_del {} {
                                portal::element::move_to_page \
                                    -page_id $page_id \
                                    -element_id $element_id \
                                    -region 1
                            }

                            portal::page::delete -page_id $max_page_id
                        }
                    }

                    # now that they have the same number of pages, get to it
                    foreach source_page_id [get_page_list -portal_id $template_id] {

                        db_1row revert_get_source_page_info {}

                        set target_page_id [db_string revert_get_target_page_id {}]

                        db_dml revert_page_update {}

                        # revert elements in two steps like "swap"
                        db_foreach revert_get_source_elements {} {
                            # the element might not be on the target page...
                            set target_element_id \
                                    [db_string revert_get_target_element {}]

                            db_dml revert_element_update {}
                        }
                    }
                }
            }
            Rename {
                portal::set_name \
                    -portal_id $portal_id \
                    -name [ns_set get $form new_name]
            }
            swap -
            move {
                portal::element::move \
                    -element_id [ns_set get $form element_id] \
                    -direction [ns_set get $form direction]
            }
            "Show Here" {
                db_transaction {
                    portal::element::move_to_page \
                        -page_id [ns_set get $form page_id] \
                        -element_id [ns_set get $form element_id] \
                        -region 1
                    portal::element::set_state \
                        -element_id [ns_set get $form element_id] \
                        -state full
                }
            }
            "Move to page" {
                portal::element::move_to_page \
                    -page_id [ns_set get $form page_id] \
                    -element_id [ns_set get $form element_id] \
                    -region 1
            }
            hide {
                set element_id_list [list]

                # iterate through the set, destructive!
                while { [expr [ns_set find $form "element_id"] + 1 ] } {
                    lappend element_id_list [ns_set get $form "element_id"]
                    ns_set delkey $form "element_id"
                }

                if {! [empty_string_p $element_id_list] } {
                    db_transaction {
                        foreach element_id $element_id_list {
                            db_dml hide_update {}

                            # after hiding an element, add
                            # it to the _first_ page
                            # of the portal.
                            portal::element::move_to_page \
                                -page_id [portal::page::get_id -portal_id $portal_id] \
                                -element_id $element_id
                        }
                    }
                }
            }
            "Change Theme" {
                set_theme_id -portal_id $portal_id -theme_id [ns_set get $form theme_id]
            }
            "Add Page" {
                set name [ns_set get $form name]
                if {[empty_string_p $name]} {
                    ad_return_complaint 1 "You must enter a name for the new page."
                }
                portal::page::new -portal_id $portal_id -name $name
            }
            "Remove Empty Page" {
                portal::page::delete -page_id [ns_set get $form page_id]
            }
            "Change Page Layout" {
                portal::page::set_layout_id \
                    -page_id [ns_set get $form page_id] \
                    -layout_id [ns_set get $form layout_id]
            }
            "Rename Page" {
                set name [ns_set get $form name]
                set page_id [ns_set get $form page_id]

                if {[empty_string_p $name]} {
                    ad_return_complaint 1 "You must enter new name for the page."
                }
                portal::page::set_name -page_id $page_id -name $name
            }
            toggle_pinned {
                set element_id [ns_set get $form element_id]

                if {[string equal [portal::element::get_state -element_id $element_id] full]} {
                    portal::element::pin -element_id $element_id
                } else {
                    portal::element::unpin -element_id $element_id
                }
            }
            toggle_shadeable {
                set element_id [ns_set get $form element_id]

                if {[string equal [portal::element::get_shadeable_p -element_id $element_id] f]} {
                    portal::element::set_shadeable_p -element_id $element_id -shadeable_p t
                } else {
                    portal::element::set_shadeable_p -element_id $element_id -shadeable_p f
                }
            }
            toggle_hideable {
                set element_id [ns_set get $form element_id]

                if {[string equal [portal::element::get_hideable_p -element_id $element_id] f]} {
                    portal::element::set_hideable_p -element_id $element_id -hideable_p t
                } else {
                    portal::element::set_hideable_p -element_id $element_id -hideable_p f
                }
            }
            default {
                ns_log error  "portal::configure_dispatch: bad op = $op!"
                ad_return_complaint 1 "portal::configure_dispatch: bad op!\nop = $op"
            }
        }
    }

    #
    # portal template procs - util and configuration
    #

    ad_proc -public template_configure {
        portal_id
        return_url
    } {
        Just a wrapper for the configure proc.

        @param portal_id
        @return A portal configuration page
    } {
        portal::configure -template_p "t" $portal_id $return_url
    }

    ad_proc -public template_configure_dispatch {
        portal_id
        form
    } {
        Just a wrapper for the configure_dispatch proc

        @param portal_id
        @param formdata an ns_set with all the formdata
    } {
        configure_dispatch -template_p "t" $portal_id $form
    }

    ad_proc -public navbar {
        {-portal_id:required}
        {-td_align "left"}
        {-link ""}
        {-pre_html ""}
        {-post_html ""}
        {-link_all 0}
        {-extra_td_html ""}
        {-table_html_args ""}
    } {
        wraps portal::dimensional to create a dotlrn navbar

        DRB: I'm only keeping this POS around for .LRN ... the portal package itself
        builds a multirow navbar suitable for use with the templating system.

        @return the id of the page
        @param portal_id
        @param link the relative link to set for hrefs
        @param current_page_link f means that there is no link for the current page
    } {
        set ad_dim_struct [list]

        db_foreach list_page_nums_select {} {
            lappend ad_dim_struct [list $page_num $name [list]]
        }

        set ad_dim_struct "{ page_num \"Page:\" 0 [list $ad_dim_struct] }"

        return [dimensional -no_header \
                -no_bars \
                -link_all $link_all \
                -td_align $td_align \
                -pre_html $pre_html \
                -post_html $post_html \
                -extra_td_html $extra_td_html \
                -table_html_args $table_html_args \
                $ad_dim_struct \
                $link]
    }

    ad_proc -private generate_action_string {
    } {
        Portal configuration pages need this to set up
        the target for the generated links. It's just the
        current location with "-2" appended to the name of the
        page.
    } {
        return "[lindex [ns_conn urlv] [expr [ns_conn urlc] - 1]]-2"
    }

    ad_proc dimensional {
        {-no_header:boolean}
        {-no_bars:boolean}
        {-link_all 0}
        {-th_bgcolor "#ECECEC"}
        {-td_align "center"}
        {-extra_td_html ""}
        {-table_html_args "border=0 cellspacing=0 cellpadding=3 width=100%"}
        {-pre_html ""}
        {-post_html ""}
        option_list
        {url {}}
        {options_set ""}
        {optionstype url}
    } {
        An enhanced ad_dimensional. see that proc for usage details

    } {
        if {[empty_string_p $option_list]} {
            return
        }

        if {[empty_string_p $options_set]} {
            set options_set [ns_getform]
        }

        if {[empty_string_p $url]} {
            set url [ad_conn url]
        }

        set html "\n<table $table_html_args>\n <tr>\n"

        if {!$no_header_p} {
            foreach option $option_list {
                append html "    <th bgcolor=\"$th_bgcolor\">[lindex $option 1]</th>\n"
            }
        }

        append html "  </tr>\n  <tr>\n"

        foreach option $option_list {
            append html "    <td align=$td_align>"

            if {!$no_bars_p} {
                append html "\["
            }

            # find out what the current option value is.
            # check if a default is set otherwise the first value is used
            set option_key [lindex $option 0]
            set option_val [lindex $option 2]
            if {![empty_string_p $options_set]} {
                set option_val [ns_set get $options_set $option_key]
            }

            set first_p 1
            foreach option_value [lindex $option 3] {
                set thisoption_name [lindex $option_value 0]
                set thisoption_value [lindex $option_value 1]
                set thisoption_link_p 1
                if {[llength $option_value] > 3} {
                    set thisoption_link_p [lindex $option_value 3]
                }

                if {$first_p} {
                    set first_p 0
                } else {
                    if {!$no_bars_p} {
                        append html " | "
                    } else {
                        append html " &nbsp; "
                    }
                }

                if {([string equal $option_val $thisoption_name] == 1 && !$link_all) || !$thisoption_link_p} {
                    append html "${pre_html}<strong>${thisoption_value}</strong>${post_html}"
                } else {
                    append html "<a href=\"$url?[export_ns_set_vars url $option_key $options_set]&[ns_urlencode $option_key]=[ns_urlencode $thisoption_name]\">${pre_html}${thisoption_value}${post_html}</a>"
                }
            }

            if {!$no_bars_p} {
                append html "\]"
            }

            append html "$extra_td_html</td>\n"
        }

        append html "  </tr>\n</table>\n"
    }

    ad_proc datasource_list {
    } {
        Returns a list of portal datasources which have been mapped to the given portal
        package instance.
    } {
        set package_id [ad_conn package_id]
        return [db_list_of_lists select_datasources {}]
    }

    ad_proc -public initialized {
        -package_id
    } {
        @param package_id The package_id of this portal instance (default ad_conn package_id)
        @return True if we've already initialized this instance of the portal package.
    } {
        return [expr {![string equal [get_master_template_id -package_id $package_id] ""]}]
    }

    ad_proc -public initialize {
    } {
        Initialize this instance of the portal package if we've not already done so.  This
        consists of creating the master template, which is assigned to party 0 and will be
        the portal returned to users who aren't logged in, or all users if the portal package
        is configured to disallow personal portals.

        The admin portal is created with our package_id as the owner.
    } {
        if { ![initialized] } {
     
            # create the master template

            portal::new \
                -name "Shared Portal" \
                -owner_id 0 \
                -owner_privileges {} \
     
            # create the admin template

            portal::new \
                -name "Portal Administration" \
                -owner_id [ad_conn package_id] \
                -theme_name blank \
                -owner_privileges {}
    
        }
    }

}
