#
#  Copyright (C) 2001, 2002 MIT
#
#  This file is part of dotLRN.
#
#  dotLRN is free software; you can redistribute it and/or modify it under the
#  terms of the GNU General Public License as published by the Free Software
#  Foundation; either version 2 of the License, or (at your option) any later
#  version.
#
#  dotLRN is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#

ad_library {

    portal element procs

    @author <a href="mailto:arjun@openforce.net">arjun@openforce.net</a>
    @author <a href="mailto:yon@openforce.net">yon@openforce.net</a>
    @creation-date 2001-09-10
    @version $Id: element-procs.tcl,v 1.12 2004/03/14 01:16:31 donb Exp $

}

namespace eval portal::element {

    ad_proc -public new {
        {-portal_id:required}
        {-page_name ""}
        {-region ""}
        {-state full}
        {-datasource_name:required}
        {-name ""}
    } {
        Create a new portal element of type datasource_name on the given portal.

        @param portal_id The portal instance to add this element to.
        @param page_name The name of the page to place it on (defaults to the first page.)
        @param pretty_name The pretty name or message resource for this element.
        @param region The page region to placer the element on.
        @param datasource_name The portal datasource name
        @param name The name of the element

        @return The element_id of the new portlet

    } {

        set page_id [portal::page::get_id -portal_id $portal_id -page_name $page_name]

        if {[string equal "" $region]} {
                set region [choose_region -page_id $page_id]
        }

        set datasource_id [portal::datasource::get_id -name $datasource_name]
        array set datasource [portal::datasource::get -datasource_id $datasource_id]

        if {[string match "" $name]} {
            set name $datasource(pretty_name)
        }

        set element_id [db_nextval portal_seq]

        db_transaction {
            if {[db_0or1row select_template_element_id {}]} {
                db_dml insert_element_with_template {}
                db_dml insert_element_parameters_with_template {}
            } else {
                db_dml insert_element {}
                db_dml insert_element_parameters {}
            }
        }

        return $element_id

    }

    ad_proc -public initialize {
        {-datasource_name:required}
        {-element_id:required}
    } {
        portal::datasource::call \
            -name $datasource_name \
            -op Initialize \
            -list_args [list $element_id]
    }

    ad_proc -public delete {
        {-element_id:required}
    } {
        db_dml delete_element {}
    }

    ad_proc -public delete_all {
        {-portal_id:required}
        {-datasource_name:required}
    } {
        delete al portal elements of type datasource_name from portal_id
    } {

        foreach element_id [get_list_from_datasource_name \
            -portal_id $portal_id \
            -datasource_name $datasource_name \
        ] {
            delete -element_id $element_id
        }

    }

    ad_proc -public get {
        {-element_id:required}
    } {
        return element info in "array get" format
    } {
        db_1row select_element {} -column_array element
        return [array get element]
    }

    ad_proc -public set_values {
        {-element:required}
    } {
        set the fields of a portal element
    } {
        array set element_array $element
        template::util::array_to_vars element_array

        db_dml update_element {}
    }

    ad_proc -public get_name {
        {-element_id:required}
    } {
        array set element [get -element_id $element_id]
        return $element(name)
    }

    ad_proc -public set_name {
        {-element_id:required}
        {-name:required}
    } {
        array set element [get -element_id $element_id]
        set element(name) $name

        set_values -element [array get element]
    }

    ad_proc -public get_page_id {
        {-element_id:required}
    } {
        return the element's page_id
    } {
        array set element [get -element_id $element_id]
        return $element(page_id)
    }

    ad_proc -public get_region {
        {-element_id:required}
    } {
        return the element's region
    } {
        array set element [get -element_id $element_id]
        return $element(region)
    }

    ad_proc -public set_region {
        {-element_id:required}
        {-region:required}
    } {
        array set element [get -element_id $element_id]
        set element(region) $region

        set_values -element [array get element]
    }

    ad_proc -public get_state {
        {-element_id:required}
    } {
        return the element's state
    } {
        array set element [get -element_id $element_id]
        return $element(state)
    }

    ad_proc -public set_state {
        {-element_id:required}
        {-state:required}
    } {
        array set element [get -element_id $element_id]
        set element(state) $state

        set_values -element [array get element]
    }

    ad_proc -public get_shadeable_p {
        {-element_id:required}
    } {
        return the element's shadeable_p
    } {
        array set element [get -element_id $element_id]
        return $element(shadeable_p)
    }

    ad_proc -public set_shadeable_p {
        {-element_id:required}
        {-shadeable_p:required}
    } {
        array set element [get -element_id $element_id]
        set element(shadeable_p) $shadeable_p

        set_values -element [array get element]
    }

    ad_proc -public get_hideable_p {
        {-element_id:required}
    } {
        return the element's hideable_p
    } {
        array set element [get -element_id $element_id]
        return $element(hideable_p)
    }

    ad_proc -public set_hideable_p {
        {-element_id:required}
        {-hideable_p:required}
    } {
        array set element [get -element_id $element_id]
        set element(hideable_p) $hideable_p

        set_values -element [array get element]
    }

    ad_proc -public pin {
        {-element_id:required}
    } {
        array set element [get -element_id $element_id]

        set element(state) pinned
        set element(shadeable_p) f
        set element(hideable_p) f

        set_values -element [array get element]
    }

    ad_proc -public unpin {
        {-element_id:required}
    } {
        set_state -element_id $element_id -state full
    }

    ad_proc -public get_list_from_datasource_name {
        {-portal_id:required}
        {-datasource_name:required}
    } {
        get a list of element_ids of type datasource_name on a particular portal
    } {
        set datasource_id [portal::datasource::get_id -name $datasource_name]
        return [db_list select_element_ids {}]
    }

    ad_proc -private move {
        {-element_id:required}
        {-direction:required}
    } {
        moves an element within a page
    } {
        if {[string equal $direction up] || [string equal $direction down]} {
            move_vertically -element_id $element_id -direction $direction
        } elseif {[string equal $direction left] || [string equal $direction right]} {
            move_horizontally -element_id $element_id -direction $direction
        }
    }

    ad_proc -private move_vertically {
        {-element_id:required}
        {-direction:required}
    } {
        swaps the element with either the previous or next one in the region,
        depending on the value of direction.
    } {

        array set element [get -element_id $element_id]
        template::util::array_to_vars element

        if {[string equal $direction up]} {
            if {![db_0or1row select_previous_element {}]} {
                return
            }
        } elseif {[string equal $direction down]} {
            if {![db_0or1row select_next_element {}]} {
                return
            }
        } else {
            ad_return_complaint 1 "portal::element::swap: bad direction: $direction"
        }

        db_transaction {
            # because of the uniqueness constraint on sort_keys we need to set
            # a dummy key, then do the swap.
            set dummy_sort_key -1

            # set the source element to the dummy key
            db_dml swap_sort_keys_1 {}

            # set the target's sort_key to the source's sort_key
            db_dml swap_sort_keys_2 {}

            # set the source's sort_key to the target's sort_key
            db_dml swap_sort_keys_3 {}
        }

    }

    ad_proc -private move_horizontally {
        {-element_id:required}
        {-direction:required}
    } {
        move a portal element between regions
    } {

        array set element [get -element_id $element_id]
        template::util::array_to_vars element

        if {[string equal $direction left]} {
            incr region -1
        } elseif {[string equal $direction right]} {
            incr region 1
        }

        db_dml update_region {}

    }

    ad_proc -private move_to_page {
        {-element_id:required}
        {-page_id:required}
        {-region ""}
    } {
        move this element to another page
    } {

        set current_region $region
        if {[empty_string_p $region]} {
            set current_region [get_region -element_id $element_id]
        }

        set target_n_regions [portal::layout::get_region_count_not_cached \
            -layout_id [portal::page::get_layout_id -page_id $page_id]
        ]

        if {$current_region > $target_n_regions} {
            set region $target_n_regions
        } else {
            set region $current_region
        }

        db_dml update_element {}

    }

    ad_proc -public choose_region {
        {-page_id:required}
    } {
        select the region on the page with the fewest elements on it
    } {

        set min_num 99999
        set min_region 0

        set layout_id [portal::page::get_layout_id -page_id $page_id]
        set region_list [portal::layout::get_region_list -layout_id $layout_id]

        foreach region $region_list {

            set count [db_string select_region_count {}]

            if {$count == 0} {
                set min_region $region
                break
            }

            if {$min_num > $count} {
                set min_num $count
                set min_region $region
            }

        }

        if {$min_region == 0} {
            set min_region 1
        }

        return $min_region

    }

    ad_proc -private get_render_data {
        {-element_id:required}
        {-edit_p f}
    } {
        Return all the good stuff a render template needs to render an element.

        @element_id The element in question
        @edit_p If true the render template should show the shade and hide widgets

    } {

        array set element [get -element_id $element_id]

        # Eventually we should be able to theme down to the element level, to allow, for
        # instance, the common undecorated center column flanked by decorated left and
        # right column format without kludging a special layout.

        array set theme [portal::theme::get \
            -theme_id [portal::get_theme_id \
                -portal_id [portal::page::get_portal_id -page_id $element(page_id)] \
            ] \
        ]
        set element(filename) "/packages/portal/lib/$theme(filename)"
        set element(resource_dir) "/resources/portal/$theme(resource_dir)"

        if {!$edit_p} {
            set element(shadeable_p) f
            set element(hideable_p) f
        }
        set element(shaded_p) [ad_decode $element(state) shaded t f]

        set config [list \
            shaded_p $element(shaded_p) \
            shadeable_p $element(shadeable_p) \
            hideable_p $element(hideable_p) \
        ]
 
        set config [concat $config [portal::element::parameter::get_all -element_id $element_id]]

        array set datasource [portal::datasource::get -datasource_id $element(datasource_id)]

        # if the element's name is the same as its datasource's, then we use
        # the datasource's pretty_name value
        if {[string equal $element(name) $datasource(name)]} {
            set element(name) $datasource(pretty_name)
        }

        set element(template_path) "/packages/$datasource(owner)/www/$datasource(template)"
        set element(config) $config

        return [array get element]

    }

    ad_proc -public configure {
        {-element_id:required}
        {-op:required}
        {-return_url:required}
    } {
        dispatch on the element_id and op requested
    } {

        set state [portal::element::get_state -element_id $element_id]

        switch $op {
            shade {
                if {[string equal $state shaded]} {
                    set new_state full
                } else {
                    set new_state shaded
                }
            }
            hide {
                if {[string equal $state hidden]} {
                    set new_state full
                } else {
                    set new_state hidden
                }
            }
        }

        portal::element::set_state -element_id $element_id -state $new_state
        ad_returnredirect $return_url

    }

# DRB: this goes away entirely ... need to do the parameter stuff though, through a 
# sevice contract call to set the portlet-specific parameters.
    ad_proc -public add_parameters {
        {-portal_id:required}
        {-portlet_name:required}
        {-value:required}
        {-key "package_id"}
        {-page_name ""}
        {-pretty_name ""}
        {-extra_params ""}
        {-force_region ""}
        {-param_action "overwrite"}
    } {
        A helper proc for portlet "add_self_to_page" procs.
        Adds the given portlet as an portal element to the given
        page. If the portlet is already in the given portal page,
        it appends the value to the element's parameters with the
        given key. Returns the element_id used.

        IMPROVE ME: refactor

        @return element_id The new element's id
        @param portal_id The page to add the portlet to
        @param portlet_name The name of the portlet to add
        @param key the key for the value (defaults to package_id)
        @param value the value of the key
        @param extra_params a list of extra key/value pairs to insert or append
    } {

        # Find out if this portlet already exists in this page
        set element_id_list [get_list_from_datasource_name -portal_id $portal_id -datasource_name $portlet_name]

        if {[llength $element_id_list] == 0} {

            db_transaction {

                # Tell portal to add this element to the page

                set element_id [new \
                        -page_id $page_id \
                        -region $force_region \
                        -datasource_name $portlet_name \
                        -name $pretty_name]

                portal::element::parameter::set_value -element_id $element_id -key $key -value $value

                if {![empty_string_p $extra_params]} {

                    for {set x 0} {$x < [llength $extra_params]} {incr x 2} {
                        portal::element::parameter::set_value \
                            -element_id $element_id \
                            -key [lindex $extra_params $x] \
                            -value [lindex $extra_params [expr $x + 1]]
                    }
                }
            }
        } else {
            db_transaction {
                set element_id [lindex $element_id_list 0]

                if {[string equal $param_action "append"]} {
                    portal::element::parameter::new -element_id $element_id -key $key -value $value
                } elseif {[string equal $param_action "overwrite"]} {
                    portal::element::parameter::set_value -element_id $element_id -key $key -value $value
                } else {
                    error "portal::add_parameters error: bad param action! $param_action 1"
                }

                if {![empty_string_p $extra_params]} {

                    for {set x 0} {$x < [llength $extra_params]} {incr x 2} {
                        if {[string equal $param_action "append"]} {
                            portal::element::parameter::new \
                                -element_id $element_id \
                                -key [lindex $extra_params $x] \
                                -value [lindex $extra_params [expr $x + 1]]
                        } elseif {[string equal $param_action "overwrite"]} {
                            portal::element::parameter::set_value \
                                -element_id $element_id \
                                -key [lindex $extra_params $x] \
                                -value [lindex $extra_params [expr $x + 1]]
                        } else {
                            error "portal::add_parameters error: bad param action! $param_action 2"
                        }
                    }
                }
            }
        }
        return $element_id
    }

    ad_proc -public remove_parameters {
        {-portal_id:required}
        {-portlet_name:required}
        {-value:required}
        {-key "package_id"}
        {-extra_params ""}
    } {
        A helper proc for portlet "remove_self_from_page" procs.
        The inverse of the above proc.

        Removes the given parameters from all the the portlets
        of this type on the given page. If by removing this param,
        there are no more params (say instace_id's) of this type,
        that means that the portlet has become empty and can be

        @param portal_id The portal page to act on
        @param portlet_name The name of the portlet to (maybe) remove
        @param key the key for the value (defaults to package_id)
        @param value the value of the key
        @param extra_params a list of extra key/value pairs to remove
    } {
        # get the element IDs (could be more than one!)
        set element_ids [get_list_from_datasource_name -portal_id $portal_id -datasource_name $portlet_name]

        # step 1: remove all the given param(s) from all of the pe's
        db_transaction {
            foreach element_id $element_ids {

                portal::element::parameter::delete \
                    -element_id $element_id \
                    -key $key \
                    -value $value

                if {![empty_string_p $extra_params]} {

                    for {set x 0} {$x < [llength $extra_params]} {incr x 2} {

                        portal::element::parameter::delete \
                            -element_id $element_id \
                            -key [lindex $extra_params $x] \
                            -value [lindex $extra_params [expr $x + 1]]
                    }
                }
            }
        }

        # step 2: Check if we should really remove the element
        db_transaction {
            foreach element_id $element_ids {
                if {[llength [portal::element::parameter::get \
                        -element_id $element_id \
                        -key $key]] == 0} {
                    remove_element -element_id $element_id
                }
            }
        }
    }

}
