#
#  Copyright (C) 2001, 2002 MIT
#
#  This file is part of dotLRN.
#
#  dotLRN is free software; you can redistribute it and/or modify it under the
#  terms of the GNU General Public License as published by the Free Software
#  Foundation; either version 2 of the License, or (at your option) any later
#  version.
#
#  dotLRN is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#

ad_library {

    portal datasource procs

    @author <a href="mailto:arjun@openforce.net">arjun@openforce.net</a>
    @author <a href="mailto:yon@openforce.net">yon@openforce.net</a>
    @creation-date 2001-09-10
    @version $Id: datasource-procs.tcl,v 1.6 2004/03/13 00:40:26 donb Exp $

}

namespace eval portal::datasource {

    ad_proc new {
        -spec:required
    } {

        Create a new portal datasource from a specification.  Why use this to define
        your portal datasources?  Because calling this procedure from your package's
        post-install procedure is easier than writing PL/SQL for Oracle and PL/pgSQL
        for Oracle.

        @param spec The specification (format described below)

        @author Don Baccus (dhogaza@pacifier.com)
        @see acs_sc::impl::new_from_spec

        The specification is a list of name-value pairs.  Possible names are

        name            The name of the new datasource
        pretty_name     The pretty name (or message key) of the datasource
        owner           The package that owns this portlet
        application     The package the portlet works with, if any
        template        The template the displays the portlet content
        admin_p         If true, this datasource is only available to admins (default is false)
        shadeable_p     If true (default) the user can shade (i.e. close) the portlet
        hideable_p      If true (default), the user can hide the portlet entirely
        description     A human-readable description (defaults to name)
        params          A list of portlet-specific param key/attributes and their values
        aliases         Service contract aliases clause for acs_sc::impl::new_from_spec

        Each parameter key can be followed by a comma-separated list of attributes
        in the familiar style of ad_page_contract or ad_form.  Do not include spaces
        in the list of attributes.  Only two attributes are allowed - "config_required_p"
        and "configured_p".

        See the portal package documentation for the meaning of these two attributes.

    } {

        # Set some default values

        set datasource(admin_p) f
        set datasource(shadeable_p) t
        set datasource(hideable_p) t

        array set datasource $spec

        # Default datasource description to its name
        if { ![info exists datasource(description)] } {
            set datasource(description) $datasource(name)
        }

        db_transaction {

        set var_list [list \
            [list name $datasource(name)] \
            [list pretty_name $datasource(pretty_name)] \
            [list owner $datasource(owner)] \
            [list application $datasource(application)] \
            [list template $datasource(template)] \
            [list admin_p $datasource(admin_p)] \
            [list shadeable_p $datasource(shadeable_p)] \
            [list hideable_p $datasource(hideable_p)] \
            [list description $datasource(description)]]

            set datasource_id [package_exec_plsql -var_list $var_list portal_datasource new]

            foreach {param value} $datasource(params) {

                if { ![regexp {^([^ \t:]+)(?::([a-zA-Z0-9_,(|)]*))} \
                           $param match param_name flags] } {
                    ad_return -code error "Parameter name '$param' doesn't have the right format. It must be var\[:flag\[,flag ...\]\]"
                }

                # set defaults for attributes
                set config_required_p f
                set configured_p f

                # now set the parameter flags
                foreach flag [split [string tolower $flags] ","] {
                    switch -exact $flag {
                        configured { set configured_p t }
                        config_required { set config_required_p t}
                        default { ad_return -code error "\"$flag\" is not a legal portal datasource attribute" }
                    }
                }

                # and define the parameter
                set_def_param -datasource_id $datasource_id \
                    -config_required_p $config_required_p \
                    -configured_p $configured_p \
                    -key $param_name \
                    -value $value

            }

            acs_sc::impl::new_from_spec \
                -spec [list name $datasource(name) \
                            contract_name portal_datasource \
                            owner $datasource(owner) \
                            aliases $datasource(aliases)]

        }
    }

    ad_proc -private set_def_param {
        {-datasource_id:required}
        {-config_required_p:required}
        {-configured_p:required}
        {-key:required}
        {-value:required}
    } {

        @author Simon Carstensen (simon@bcuni.net)

    } {
        set var_list [list \
            [list datasource_id $datasource_id] \
            [list config_required_p $config_required_p] \
            [list configured_p $configured_p] \
            [list key $key] \
            [list value $value]]

        package_exec_plsql -var_list $var_list portal_datasource set_def_param
    }

    ad_proc -private delete {
        {-name:required}
    } {

        @author Don Baccus (dhogaza@pacifier.com)

    } {

        if { ![db_0or1row get_datasource_id {}] } {
            # Returning an error here is a PITA developing, so we'll just log an error
            ns_log Error "Datasource \"$name\" does not exist"
            return
        }

        acs_sc::impl::delete -contract_name portal_datasource -impl_name $name
        set var_list [list [list datasource_id $datasource_id]]
        return [package_exec_plsql -var_list $var_list portal_datasource del]
    }


    ad_proc -public get {
        {-datasource_id:required}
    } {
        return the datasource in "array get" format
    } {
        db_1row select_datasource {} -column_array datasource
        return [array get datasource]
    }

    ad_proc -public get_name {
        {-datasource_id:required}
    } {
        array set datasource [get -datasource_id $datasource_id]
        return $datasource(name)
    }

    ad_proc -public get_id {
        {-name:required}
    } {
        get the id from the name
    } {
        return [db_string select_id {}]
    }

    ad_proc -public call {
        {-name:required}
        {-op:required}
        {-list_args ""}
    } {
        call a method on a datasource
    } {
        return [acs_sc_call portal_datasource $op $list_args $name]
    }

}
