namespace eval phb {
    namespace eval form {}
    namespace eval validate {}
}

ad_proc -public phb::package_url {} {
    returns the package url for the mounted instance of 
    photobook.
} { 
    return [apm_package_url_from_key photobook]
}

ad_proc -public phb::root_folder {} {
    returns the root folder for the mounted instance of 
    photobook.
} { 
    return [parameter::get -parameter RootFolder -package_id [apm_package_id_from_key photobook]]
}

ad_proc -private phb::privacy_p {} {
    Is acceptance of a privacy policy required?
} { 
    return [parameter::get -parameter RequirePolicyP -package_id [apm_package_id_from_key photobook]]
}


ad_proc -public phb::key { user_id } {
    returns the cr_items.name key for a given user.
} { 
    return phb_$user_id
} 

ad_proc -private phb::options_no_cache { key } {
    returns a list of lists from the phb_backing_data table.

    call the caching version!
} {
    set ret {}
    
    db_foreach select_keys {
        SELECT short_name as value,short_name as key 
        FROM phb_backing_data 
        WHERE data_type = :key 
        ORDER by seq, short_name
    } { 
        lappend ret [list $value [ad_quotehtml $key]]
    }
    
    return $ret
}

ad_proc -public phb::options {
    -key
    {-prefix {{-- {}}}}
    {-postfix {}}
} {
    Returns a list of lists for a select widget.
    key is the data class to return such as degree or livinggroup.
    
    @param key lookup options for this key
    @param prefix tack the list of pairs to the front of the list
    @param postfix tack the list of pairs to the end of the list
} { 
    return [concat\
                $prefix \
                [util_memoize [list phb::options_no_cache $key]] \
                $postfix ]
} 


ad_proc -private phb::options_grouped_no_cache { key } {
    returns a list of lists from the phb_backing_data table.

    call the caching version!
} {
    set out [list]
    db_foreach select_keys {
        SELECT short_name as val,short_name as key, group_name as g
        FROM phb_backing_data 
        WHERE data_type = :key 
        ORDER by group_name, seq, short_name
    } { 
        set val "$g : $val"
        lappend out [list $val $key]
    } 
    
    return $out
} 


ad_proc -public phb::options_grouped {
    -key
    {-prefix {{-- {}}} }
    {-postfix {}}
} {
    Returns a list of lists for a grouped select widget.
    key is the data class to return such as degree or livinggroup.
    
    @param key lookup options for this key
    @param prefix tack the list of pairs to the front of the list
    @param postfix tack the list of pairs to the end of the list
} { 
    return [concat\
                $prefix \
                [util_memoize [list phb::options_grouped_no_cache $key]] \
                $postfix ]
} 

ad_proc -private phb::country_options_no_cache {} { 
    returns a list of country, iso code 
} {
    return [db_list_of_lists countries {select default_name, iso from countries}]
}

ad_proc -public phb::country_options {
    {-prefix {{-- {}}}}
    {-postfix {}}
} {
    Returns a list of lists of country data
    
    @param key lookup options for this key
    @param prefix tack the list of pairs to the front of the list
    @param postfix tack the list of pairs to the end of the list
} { 
    return [concat\
                $prefix \
                [util_memoize [list phb::country_options_no_cache]] \
                $postfix ]
} 

ad_proc -public phb::get_country_name { iso } { 
    Return the full country name given an iso 
} { 
    set iso [string toupper [string trim $iso]]

    foreach c [util_memoize [list phb::country_options_no_cache]] { 
        if {[string equal [lindex $c 1] $iso]} { 
            return [lindex $c 0]
        } 
    } 
    
    return {}
}

ad_proc -public phb::year_options {
    {-start 0}
    {-end 5}
    {-prefix {{-- {}}}}
    {-postfix {}}
} { 
    generate [list [list year year] ... ]
    starting from now + start to now + end
} { 
    set ret [list]
    set year [expr {[clock format [clock seconds] -format "%Y"] + $start}]

    if {$start > $end} { 
        set inc -1
        set comp {$i >= $end} 
    } else { 
        set inc 1
        set comp {$i <= $end} 
    }

    for {set i $start} $comp {incr i $inc} {
        lappend ret [list $year $year]
        incr year $inc
    }

    return [concat $prefix $ret $postfix ]
}

ad_proc -public phb::suffix_spacer { suffix } {
    returns an appropriate spacer for the specified suffix
} {
    if ([regexp -lineanchor -- {^.[A-Z]} $suffix]) {
        return ", "
    } else {
        return " "
    }  
}

ad_proc -public phb::private_flags_sql {
    {-prefix ""}
    object_type
} {
    generates an attribute list of privacy flags for an object type
} {
    set attrs [phb::private_flags $object_type]
    set prefix_attrs [list]

    if {[string length $prefix]} {
        append prefix "."
    } 

    foreach attr $attrs {
        lappend prefix_attrs "${prefix}$attr"
    }

    return [join $prefix_attrs ", "]
}

ad_proc -private phb::private_flags_no_cache {
    object_type
} {
    generates an attribute list of privacy flags for an object type
} {
    return [db_list attributes {
        SELECT attribute_name
          FROM acs_attributes
         WHERE object_type = :object_type
           and attribute_name like 'priv%'
    }]
}

ad_proc -private phb::type_attributes_no_cache { 
    object_type
} {
    generates an attribute list for an object type
} {
    return [db_list attributes {
        SELECT attribute_name
          FROM acs_attributes
         WHERE object_type = :object_type
    }]
}

ad_proc -private phb::type_attributes { 
    object_type
} {
    generates an attribute list for an object type
} {
    return [util_memoize [list phb::type_attributes_no_cache $object_type]]
}



ad_proc -private phb::private_flags { object_type } { 
    Return a list of privacy fields 
} { 
    return [util_memoize [list phb::private_flags_no_cache $object_type]]
} 

ad_proc -public phb::mask_private_attrs {
    {-override 0}
    {-excludes {}}
    object_array
    private_attrs
} {
    blanks any attributes in the object array that are private 
} {
    upvar $object_array object 

    foreach mask $private_attrs {

        if {[empty_string_p $object($mask)]} { 
            set object($mask) 0
        } 

        if { [expr {$override + $object($mask) > 0}] }  {
            regsub {priv_?} $mask {} mask
            set attrs [array names object "${mask}*"]

            foreach attr $attrs {
                if {[lsearch $excludes $attr] == -1 
                    && ![string match priv* $attr]} {
                    set object($attr) ""
                } 
            }
        }
    }
}


ad_proc -private phb::form::field_map {
    relation
} { 
    returns a list of fields and metadata for a given relation type
    each field has a structure \{field dummy type class\} for admin
    each field has a structure \{field type dummy class\} otherwise
    class can be a list 
    the metadata can be a list or a simple string.
    key:
    a -- means Contact Information
    d -- means displays
    e -- means Educational history
    h -- means hidden?
    m -- means Undergraduates
    p -- means Personal Data 
    x -- means Exclude the field
    z -- means Admin view
    flags in class may have different meaning than type and dummy.
    @see phb::form::excludes
} { 
    switch $relation { 
        person { 
            return { 
                { user_id h h {a e p m z} }
                { institute_id x e {p z} } 
                { priv x x {} } 
                { salutation e e p } 
                { first_name d e {p z} } 
                { middle_name d e {p z} }
                { priv_middle_name x x {} }
                { last_name d e {p z} }
                { suffix e e {p z} }
                { preferred_name e e p }
                { priv_preferred_name x x {} }
                { former_name e e {p z} }
                { priv_former_name x x {} }
                { gender e e {p z} }
                { priv_gender x x {} }
                { birthdate e e {p z} }
                { priv_birthdate x x {} }
                { ethnicity x e {p z} }
                { priv_ethnicity x x {} }
                { email_outside x x {a z} }
                { priv_email_outside x x {} }
                { email_efl x x  {a z} }
                { priv_email_efl x x {} }
                { email_primary x x  {a z} }
                { priv_email_primary x x {} }
                { program d e e }
                { priv_program x x {} }
                { class_year d e e }
                { priv_class_year x x {} }
                { preferred_graduation e e e }
                { expected_graduation d e e }
                { living_group e e a}
                { priv_living_group x x {} }
                { urop_1 e e {e m}}
                { urop_2 e e {e m}}
                { priv_urop x x {} }
                { concentration_1 e e e }
                { concentration_2 e e e }
                { priv_concentration x x {} }
                { academic_interest_1 e e e }
                { academic_interest_2 e e e }
                { academic_interest_3 e e e }
                { priv_academic_interest x x {} }
                { outside_interest_1 e e p }
                { outside_interest_2 e e p }
                { outside_interest_3 e e p }
                { priv_outside_interest x x {} }
                { hometown_city x x {} }
                { priv_hometown_city x x {} }
                { hometown_state x x {} }
                { priv_hometown_state x x {} }
                { hometown_postcode x x {} }
                { priv_hometown_postcode x x {} }
                { hometown_country x x {} }
                { priv_hometown_country x x {} }
                { priv_hometown x x {} }
                { origin e e p}
                { priv_origin x x {} }
                { citizenship e e p }
                { priv_citizenship x x {} }
                { marital_status e e p }
                { priv_marital_status x x {} }
                { partner_firstname e e p}
                { partner_lastname e e p}
                { priv_partner x x {} }
                { child_1 e e p}
                { child_born_1 e e p}
                { child_2 e e p}
                { child_born_2 e e p}
                { child_3 e e p}
                { child_born_3 e e p}
                { child_4 e e p}
                { child_born_4 e e p}
                { child_5 e e p}
                { child_born_5 e e p}
                { priv_child x x {} }
                { favorite_place e e p}
                { priv_favorite_place x x {} }
                { favorite_book e e p}
                { priv_favorite_book x x {} }
                { favorite_movie e e p}
                { priv_favorite_movie x x {} }
                { one_word_description e e p}
                { priv_one_word x x {} }
                { language_1 e e p}
                { language_2 e e p}
                { language_3 e e p}
                { priv_language x x {} }
                { priv_portrait x x {} }
                { past_employers e e m }
                { status x e {p z} }
                { status_note x e {p z} }
            }

        }
        
        internship { 
            return { 
                { priv x x {}}
                { institution e e {e m}}
                { location x x {}}
                { department x x {}}
                { relation x x {}}
                { text_date x x {}}
                { started x x {}}
                { ended x x {}}
            } 
        } 
        job_current - 
        job_past { 
            return { 
                { priv x x {}}
                { institution e e {m}}
                { location e e {m}}
                { department x x {}}
                { relation e e {m}}
                { text_date x x {}}
                { started x x {}}
                { ended x x {}}
            } 
        } 
        degree { 
            return { 
                { priv x x {}}
                { institution e e e}
                { location x x {}}
                { department e e e}
                { relation e e e}
                { text_date x x {}}
                { started e e e}
                { ended e e e}
            } 
        } 
        home { 
            return { 
                { priv x x {}}
                { other_description x x {}}
                { country_code e e a}
                { area_code e e a}
                { phone_number e e a}
                { extension e e a}
            } 
        }
        mobile { 
            return { 
                { priv x x {}}
                { other_description x x {}}
                { country_code e e a}
                { area_code e e a}
                { phone_number e e a}
                { extension x x {}}
            }
        } 
        address { 
            return { 
                { priv x x {}} 
                { address_other x x {}} 
                { address_1 e e a} 
                { address_2 e e a} 
                { address_3 e e a} 
                { priv_address x x {}} 
                { city e e a} 
                { state e e a} 
                { postcode e e a} 
                { country e e a} 
            } 
        } 
        hometown { 
            return { 
                { priv x x {}} 
                { address_other x x {}} 
                { address_1 x x {}} 
                { address_2 x x {}} 
                { address_3 x x {}} 
                { priv_address x x {}} 
                { city e e a} 
                { state e e a} 
                { postcode e e a} 
                { country e e a} 
            } 
        } 
        default { 
            return [list]
        } 
    }
} 

ad_proc -private phb::form::excludes {
    -admin:boolean
    relation ui_class
} {  
    # Generate a list of exluded fields for a given relation

    if {$admin_p} { 
        set vars {field dummy type class} 
    } else { 
        set vars {field type dummy class} 
    } 
    
    set field_list [list]
    
    foreach fields [phb::form::field_map $relation] { 
        foreach $vars $fields { break } 

        if {[string equal $type x] 
            || ( [lsearch $class $ui_class] == -1
                 && ![empty_string_p $ui_class])
        } { 
            lappend field_list $field
        } 
        
    }

    return $field_list
}

ad_proc -private phb::form::displays {
    -admin:boolean
    relation ui_class
} {  
    # Generate a list of exluded fields for a given relation

    if {$admin_p} { 
        set vars {field dummy type class} 
    } else { 
        set vars {field type dummy class} 
    } 
    
    set field_list [list]
    
    foreach fields [phb::form::field_map $relation] { 
        foreach $vars $fields { break } 

        if {[string equal $type d] 
            && ( [lsearch $class $ui_class] >  -1
                 || [empty_string_p $ui_class])
        } { 
            lappend field_list $field
        } 
        
    }

    return $field_list
}

ad_proc -private phb::form::ask {
    -admin:boolean
    relation ui_class
} { 
    # Generate a list of exluded fields for a given relation
    if {$admin_p} { 
        set vars {field dummy type class} 
    } else { 
        set vars {field type dummy class} 
    } 
    
    set use 0

    foreach fields [phb::form::field_map $relation] { 
        foreach $vars $fields { break } 

        if { ![string equal $type x] 
             && ( [empty_string_p $ui_class] 
                  || [lsearch $class $ui_class] > -1 ) } {
            set use 1
            break
        } 
    }

    ns_log Warning "JCD: $use relation $relation class $ui_class"

    return $use
}

ad_proc -public phb::form::create {
    -form_name 
    -user_id 
    -admin:boolean 
    {-form_type {}}
} { 
    creates a form for entering user data
    
    @param form_name The name of the form in the templating system
    @param user_id Create form for which users data 
    @param admin whether the form is for an admin or a regular user
    @param type the type of data to edit (in {}, a, p, e, m) 
    a=address, p=personal, e=education, m=employment.
    the empty string is for all fields.
    
    @return a fully populated form data structure.
} { 
    set parent_id [phb::root_folder]
    set base [phb::key $user_id]
    set exclude_base {title description mime_type}
    
    # Save the form_type
    template::element create $form_name form_type \
        -datatype text \
        -widget hidden \
        -value $form_type \
        -sign

    
    # Figure out if we are going to edit existing data or create new data.
    set initial_p 1

    array set child [phb::users_items -user_ids $user_id]
    if { ![empty_string_p $child(phb_person.parent)] }  { 
            set initial_p 0
    } 


    set prefixes {}
    array set section {{} "Personal data" a "Contact information" m "<center>Undergraduates</center>" e "Educational history" p "Personal data" z "Admin view"}


    # Now go through the item types and see if we need to add each one
    # for the given form_type
    
    # The main person record.  As it stands this should always be present and I 
    # believe if it is not there might be some problems.
    
    template::form section $form_name $section($form_type)
    if {$form_type == "a" || $form_type == "z"} {
        if {$admin_p} { 
            template::element create $form_name alias -datatype text -widget text -label "Sloan email alias" -optional
            template::element create $form_name efl -datatype text -widget text -label "Sloan EFL" -optional
        } else { 
            template::element create $form_name alias -datatype text -widget inform -label "Sloan email alias" 
            template::element create $form_name efl -datatype text -widget inform -label "Sloan EFL" 
        } 
        # only allow user to edit their forwarding address if they have an
        # EFL;  without one it is meaningless
        if { [string length [db_string get_efl "select efl from sloan_email where user_id = :user_id" -default ""]] > 0 } {
          template::element create $form_name outside -datatype email -widget text -label "EFL Forwarding" -validate { { expr [string first @sloan $value] == -1 } { EFL Forwarding must not be to a Sloan e-mail address } }
        } else {
          template::element create $form_name outside -datatype email -widget inform -label "EFL Forwarding"
        }
    } 
        
    if {[phb::form::ask -admin=$admin_p person $form_type]} { 
        set prefix prsn_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p person $form_type]]
        content::new_item_form -form_name $form_name \
            -content_type phb_person \
            -content_method no_content \
            -name $base \
            -item_id $child(phb_person.parent) \
            -section $section($form_type) \
            -exclude $exclude \
            -prefix $prefix \
            -parent_id $parent_id \
            -hidden {name}

        # Maybe we should check if the element exists here 
        # but if it does that would be a significant bug so 
        # throwing an error here would be a better idea.
        template::element::create $form_name user_id \
            -widget hidden \
            -datatype integer \
            -hidden \
            -value $user_id \
            -sign

        template::element::create $form_name ${prefix}user_id \
            -widget hidden \
            -datatype integer \
            -hidden \
            -value $user_id \
            -sign

        set root_id [template::element get_value $form_name ${prefix}item_id]

        foreach display_field [phb::form::displays -admin=$admin_p person $form_type] { 
            template::element::set_properties $form_name $prefix$display_field -widget inform
        } 

        if {[template::form is_request $form_name]} { 
            if {[db_0or1row name {select nvl(outside,email) as outside, alias, efl, first_names, last_name
                FROM cc_users c, sloan_email s
                WHERE c.user_id = :user_id
                and s.user_id(+) = c.user_id
            }]} { 
                if {[template::element exists $form_name ${prefix}last_name]
                    && [template::element exists $form_name ${prefix}first_name] 
                    && $initial_p
                } { 
                    template::element set_properties $form_name ${prefix}first_name value $first_names
                    template::element set_properties $form_name ${prefix}last_name value $last_name
                } 

                if { [template::element exists $form_name outside] } { 
                    template::element set_properties $form_name outside value $outside
                } 
                if { [template::element exists $form_name efl] } { 
                    template::element set_properties $form_name efl value $efl
                } 
                if { [template::element exists $form_name alias] } { 
                    template::element set_properties $form_name alias value $alias
                } 
            }
        }

    } else { 
        set root_id $child(phb_person.parent)
    } 


    if {[phb::form::ask -admin=$admin_p internship $form_type]} { 
        set prefix intn_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p internship $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_span \
            -prefix $prefix \
            -section "Internship" \
            -name "${base}_internship" \
            -relation "internship" \
            -item_id $child(phb_span.internship) \
            -content_method no_content \
            -exclude $exclude \
            -hidden {name span_type} 

        template::element set_properties $form_name ${prefix}institution label "Company"
        template::element set_properties $form_name ${prefix}span_type value internship
    } 

    if {[phb::form::ask -admin=$admin_p job_current $form_type]} { 
        # Current Employer
        set prefix cemp_
        lappend prefixes $prefix
        
        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p job_current $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_span \
            -content_method no_content \
            -prefix $prefix \
            -section "<center>Graduate students</center>Current employer" \
            -name "${base}_job_current" \
            -relation "job_current" \
            -item_id $child(phb_span.job_current) \
            -exclude $exclude \
            -hidden {name span_type} 

        template::element set_properties $form_name ${prefix}institution label "Employer"
        template::element set_properties $form_name ${prefix}relation label "Position"
        template::element set_properties $form_name ${prefix}span_type value job
    } 

    if {[phb::form::ask -admin=$admin_p job_past $form_type]} { 
        # Previous Employer
        set prefix pemp_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p job_past $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_span \
            -content_method no_content \
            -prefix $prefix \
            -section "Previous employer" \
            -name "${base}_job_past" \
            -relation "job_past" \
            -item_id $child(phb_span.job_past) \
            -exclude $exclude \
            -hidden {name span_type}

        template::element set_properties $form_name ${prefix}institution label "Employer"
        template::element set_properties $form_name ${prefix}relation label "Position"

        template::element set_properties $form_name ${prefix}span_type value job
    } 

    if {[phb::form::ask -admin=$admin_p degree $form_type]} { 
        # Past Degrees
        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p degree $form_type]]
        foreach n {1 2 3 4} {

            set prefix deg${n}_
            lappend prefixes $prefix

            content::new_item_form -form_name $form_name \
                -parent_id $root_id \
                -content_type phb_span \
                -content_method no_content \
                -prefix $prefix \
                -section "Past degree $n" \
                -name "${base}_degree_$n" \
                -relation "degree" \
                -item_id [lindex $child(phb_span.degree) [expr $n - 1]] \
                -exclude $exclude \
                -hidden {name span_type}
            
            template::element set_properties $form_name ${prefix}institution label "School/University"
            template::element set_properties $form_name ${prefix}institution widget select
            template::element set_properties $form_name ${prefix}institution html {size 1}
            template::element set_properties $form_name ${prefix}institution options [phb::options_grouped -key university]

            template::element set_properties $form_name ${prefix}department label "Major"
            template::element set_properties $form_name ${prefix}department widget select
            template::element set_properties $form_name ${prefix}department html {size 1}
            template::element set_properties $form_name ${prefix}department options [phb::options -key field -postfix {{Other Other}}]

            template::element set_properties $form_name ${prefix}relation label "Degree"
            template::element set_properties $form_name ${prefix}relation widget select
            template::element set_properties $form_name ${prefix}relation html {size 1}
            template::element set_properties $form_name ${prefix}relation options [phb::options -key degree -postfix {{Other Other}}]

            template::element set_properties $form_name ${prefix}span_type degree
        }
    } 
    
    if {[phb::form::ask -admin=$admin_p hometown $form_type]} { 

        # Contact details.
        set prefix hmtn_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p hometown $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_address \
            -content_method no_content \
            -prefix $prefix \
            -section "Hometown" \
            -name "${base}_hometown" \
            -relation "hometown" \
            -item_id $child(phb_address.hometown) \
            -exclude $exclude \
            -hidden {name}

        phb::validate::address -prefix $prefix -form $form_name
    } 

    if {[phb::form::ask -admin=$admin_p address $form_type]} { 

        # Contact details.
        set prefix cadd_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p address $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_address \
            -content_method no_content \
            -prefix $prefix \
            -section "Current contact details" \
            -name "${base}_address" \
            -relation "address" \
            -item_id $child(phb_address.address) \
            -exclude $exclude \
            -hidden {name}

        phb::validate::address -prefix $prefix -form $form_name
    } 


    if {[phb::form::ask -admin=$admin_p home $form_type]} { 
        set prefix phhm_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p home $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_phone \
            -content_method no_content \
            -prefix $prefix \
            -section "Home phone" \
            -name "${base}_home" \
            -relation "home" \
            -item_id $child(phb_phone.home) \
            -exclude $exclude \
            -hidden {name phone_type}

        if {[template::form is_request $form_name]
            && $initial_p 
        } { 
            if { [template::element exists $form_name ${prefix}phone_type] } { 
                template::element set_properties $form_name ${prefix}phone_type value home
            } 
            if { [template::element exists $form_name ${prefix}country_code] } { 
                template::element set_properties $form_name ${prefix}country_code value "+1"
            } 
        } else { 
            phb::validate::phone -prefix $prefix -form $form_name
        } 
    }

    if {[phb::form::ask -admin=$admin_p mobile $form_type]} { 
        # Mobile phone...
        set prefix phnm_
        lappend prefixes $prefix

        set exclude [concat $exclude_base [phb::form::excludes -admin=$admin_p mobile $form_type]]
        content::new_item_form -form_name $form_name \
            -parent_id $root_id \
            -content_type phb_phone \
            -content_method no_content \
            -prefix $prefix \
            -section "Mobile phone" \
            -name "${base}_mobile" \
            -relation "mobile" \
            -item_id $child(phb_phone.mobile) \
            -exclude $exclude \
            -hidden {name phone_type}

        # Cram in +1 for us country code.
        if {[template::form is_request $form_name]
            && $initial_p 
        } { 
            if { [template::element exists $form_name ${prefix}phone_type] } { 
                template::element set_properties $form_name ${prefix}phone_type value home
            } 
            if { [template::element exists $form_name ${prefix}country_code] } { 
                template::element set_properties $form_name ${prefix}country_code value "+1"
            } 
        } else { 
            phb::validate::phone -prefix $prefix -form $form_name
        } 
    } 
    
    # Clonk in the prefix list to the template
    template::element create $form_name content_prefixes \
        -datatype text \
        -widget hidden \
        -value $prefixes \
        -sign

}


ad_proc -public phb::email_available {email user_id} {
    set email [string tolower [string trim $email]]
    set email_full "$email@sloan.mit.edu"
    set root_folder [phb::root_folder]

    # We query the sed_email_syn table once per field to avoid disaster.
    # fix the view would be a better answer...

    return [db_string check_available {
        select count(*) from dual 
        where not exists (select 1 
                          from parties 
                          where email = :email_full 
                            and party_id <> :user_id) 
          and not exists (select 1 
                          from sloan_email 
                          where user_id <> :user_id
                            and ( alias = :email or efl = :email or outside = :email_full ))
          and not exists (select 1
                          from sed_email_syn 
                          where spsloanemail = :email or spalias = :email)
          and not exists (select 1
                          from SLOANMAIL_ALIASES_SYN
                             where SLOANMAIL_ALIAS = :email)
        
    }]
}


ad_proc -public phb::privacy_accepted {
    -user_id
} { 
    if {$user_id} { 
        set accepted_p {}
        db_0or1row accepted {select accepted_p from phb_privacy where user_id = :user_id}
        return $accepted_p
    } 
    return {}
}

ad_proc -public phb::privacy_set {
    -user_id 
    -accepted
} { 
    Set the privacy policy accepted to t or f.
} { 
    set ip [ad_conn peeraddr] 
    set created_by [ad_conn user_id]

    if {[catch {db_dml insert {insert into phb_privacy (
                                                        user_id,  accepted_p, creation_user, creation_date, creation_ip
                                                        ) values ( 
                                                                  :user_id, :accepted, :created_by, sysdate, :ip
                                                                  )}} errMsg]} { 
        if {[string match -nocase "*PHB_PRIVACY_USER_ID_PK*" $errMsg]} { 
            # primary key violation, update instead.
            if {[catch {db_dml update {update phb_privacy set accepted_p = :accepted, creation_user = :created_by, creation_ip = :ip, creation_date = sysdate where user_id = :user_id}} errMsg]} { 
                ns_log Error "Error updating acceptance record: $errMsg"
            } 
        } else { 
            ns_log Error "Error inserting acceptance record: $errMsg"
        } 
    } 
}



ad_proc -private phb::user_state_no_cache { user_id } {
    returns ok or an error with errorCode indicating privacy new 
    (which prevents util_memoize from caching non ok status).
} { 
    if {![string is integer $user_id] || !$user_id} { 
        return login
    } 

    if { ![phb::privacy_p] 
         || [string equal [phb::privacy_accepted -user_id $user_id] "t"] } {
        
        if {[phb::user_can_create_efl $user_id]
            && [dotlrn::user_can_browse_p -user_id $user_id]
        } { 
            set parent_id [phb::root_folder]
            set name [phb::key $user_id]
            
            if {[db_0or1row exists {select efl, alias from sloan_email where user_id = :user_id}]} { 
                if {[empty_string_p $efl] || [empty_string_p $alias]} {
                    return -code error -errorcode new -errorinfo "No sloan email defined for user $user_id"
                } 
            } else { 
                return -code error -errorcode new -errorinfo "No photobook record for user $user_id"
            } 
        }
    } else { 
        return -code error -errorcode privacy -errorinfo "User has not accepted the privacy policy"
    }
    
    return ok
}



ad_proc -public phb::user_state {{user_id {}}} { 
    check on the user state

    @return ok, new, privacy, login

    @author Jeff Davis (davis@xarg.net)
} {     
    if {[empty_string_p $user_id]} { 
        set user_id [ad_conn user_id]
    } 

    if {![string is integer $user_id] || !$user_id} { 
        return login
    } 

#    if {[catch {set code [util_memoize [list phb::user_state_no_cache $user_id]]} err]} 
    if {[catch {set code [phb::user_state_no_cache $user_id]} err]} { 
        global errorCode errorInfo
        return $errorCode
    } else { 
        return $code
    } 
}

ad_proc -public phb::dispatch {
    {-user_id {}}
    {-page {}}
} { 
    Dispatch the user to the right page for 
    their given state.
} { 
    set my_user_id [ad_conn user_id]
    if {[empty_string_p $user_id]} { 
        set user_id $my_user_id
    } 

    set state [phb::user_state $my_user_id]
    
    switch $state { 
        privacy { 
            ad_returnredirect "[phb::package_url]privacy-policy?return_url=[ad_return_url -urlencode]"
            ad_script_abort
        } 
        new { 
            if {[string equal $page edit]
                && [string equal $user_id $my_user_id]
            } { 
                ad_returnredirect "[phb::package_url]new?return_url=[ad_return_url -urlencode]"
                ad_script_abort
            } 
        } 
        login { 
            ad_redirect_for_registration
            ad_script_abort
        } 
        ok - 
        default { 
            # do nothing 
        } 
    } 

    return $my_user_id
}


proc phb::efl_token {elem} { 
    set elem [string trim [string tolower $elem]]

    set elem [string map {. { }} $elem]
    regsub {\s+} $elem { } elem
    set elem [string map {{ } _ - _ ' {} , {}} $elem]
    set elem [string trim $elem {. _-}]
    
    return $elem
}

proc phb::get_sloan_efl {first middle last user_id} {
    # lowerify last and strip the 
    regsub {,? (jr|sr|ii|iii|iv|v|vi)[.]?$} [string trim [string tolower $last]] {} last
    set last [phb::efl_token $last]
    
    set first [phb::efl_token $first]

    if {![empty_string_p $middle]} { 
        set middle [phb::efl_token $middle]
        set minit  [string index $middle 0]
    } else { 
        set minit {}
    } 

    set try [join [list $first $last] .]

    if {[phb::email_available $try $user_id]} { 
        return $try
    } 

    if {![empty_string_p $minit]} { 
        set try [join [list $first $minit $last] .]
        if {[phb::email_available $try $user_id]} { 
            return $try
        } 
        set try [join [list $first $middle $last] .]
        if {[phb::email_available $try $user_id]} { 
            return $try
        } 
    } 

    foreach i {2 3 4 5 6 7 8 9} {
        set tryn "$try$i"
        if {[phb::email_available $tryn $user_id]} {
            return $tryn
        } 
    } 
    
    error "no efl email available for $first $middle $last $user_id"
} 


ad_proc -public phb::image_information {file} { 
    if { [catch {set size [file size $file]} errMsg] } {
        return -code error $errMsg
    }
    if { [ catch {set out [exec identify -ping -format "%w %h %m" $file]} errMsg]} {
        return -code error $errMsg
    }

    foreach {width height type} [split $out { }] { break }
    switch $type {
        JPG - JPEG {
            set mime image/jpeg
        }
        GIF - GIF87 {
            set mime image/gif
        }
        default {
            set mime {}
        }
    }
    
    return [list width $width height $height mime_type $mime magictype $type file_size $size file $file]
}


ad_proc -private phb::remove_files {files} { 
    foreach file $files { 
        if {[catch {file delete -force -- $file} err]} { 
            ns_log Error "phb::remove_files $file error $err"
        } 
    } 
} 


ad_proc -public phb::generate_photos {file} { 
    # Get the image information for the base image
    set remove [list]
    set images [list]

    array set info [phb::image_information $file]

    global errorInfo
    
    if {[empty_string_p $info(mime_type)]} { 
        # If we got an image type we did not recognize try to turn it to a jpeg.

        if {[catch {exec convert $file JPG:$file.jpg} errMsg]} {
	    set error $errorInfo
            catch {file delete -force -- $file} 
            catch {file delete -force -- $file.jpg} 
            error $errMsg $error PHB_IMG_CONVERT_ERROR
        }
        
        catch {file delete -force -- $file} 
        set file "${file}.jpg"
        set info(mime_type) image/jpeg
        lappend remove $file
    } 
    
    # Create the image upload metadata and the sizes we use...
    set padding_color [parameter::get -parameter PaddingColor -default "#ffffff"]

    foreach {rel width height sizing} [parameter::get -parameter ImageSizes] { 
        if {![empty_string_p $width$height]} { 
            if {[string equal $sizing none]} { 
                set geom "${width}x${height}"                
            } else { 
                set geom "${width}x${height}!"
            } 

            set pad [phb::pad $sizing $width $height $info(width) $info(height)]
            set crop [phb::crop $sizing $width $height $info(width) $info(height)]
            ns_log Debug "JCD: was $info(width) $info(height): convert -border $pad -bordercolor $padding_color -chop $crop -geometry $geom"
            if {[catch {exec convert -border $pad -bordercolor $padding_color -shave $crop -geometry $geom -interlace None -sharpen 1x2 $file ${file}-$rel} errMsg]} { 
		set error $errorInfo
                lappend remove ${file}-$rel
                phb::remove_files $remove
		error $errMsg $error PHB_IMG_CONVERT_ERROR
            } 
            lappend remove ${file}-$rel
            lappend images [concat relation_tag $rel [phb::image_information ${file}-$rel]]
        } else { 
            lappend images [concat relation_tag $rel [phb::image_information $file]]
        } 
    } 

    return $images
}

ad_proc -private phb::crop {m w h iw ih} {
    Compute the proper crop prior to scaling 
    m in none pad crop best
    w h desired width height
    iw ih input width height

} { 
    set ar [expr {$w*1.0/$h}]
    set iar [expr {$iw*1.0/$ih}]

    # don't crop if none, pad, if too tall, or to small 
    
    if {[string equal $m none]
        || [string equal $m pad]
        || $ar == $iar
        || ([string equal $m best] && $ar > $iar ) 
        || ([string equal $m best] && $w > $iw && $h > $ih)
    } { 
        return 0x0
    }
    
    # figure out how much to clip from either side from the 
    # original image to make the aspect ratios match

    if {$ar > $iar} { 
        # ar > iar means the input image is too tall so chop of
        # height
        set crop [expr {round(($ih*1.0 - ($h * $iw * 1.0/$w))/2.0)}]

        return 0x$crop
    } else { 
        set crop [expr {round(($iw*1.0 - ($ih * $w * 1.0/$h))/2.0)}]
      
        return ${crop}x0
    } 
}


ad_proc -private phb::pad {m w h iw ih} {
    Compute the proper pad prior to scaling 
    m in none pad crop best
    w h desired width height
    iw ih input width height
} { 
    set ar [expr {$w*1.0/$h}]
    set iar [expr {$iw*1.0/$ih}]

    # don't crop if none, pad, if too tall, or to small 
    if {[string equal $m none]
        || [string equal $m crop]
        || $ar == $iar
        || ([string equal $m best] 
            && $ar < $iar 
            && ! ($iw < $w && $ih < $h))
    } { 
        return 0x0
    }
    
    # figure out how much to clip from either side from the 
    # original image to make the aspect ratios match

    if {$ar > $iar} { 
        # ar > iar means the input image is too tall so pad width
        # height
        set pad [expr {round(($w*$ih*1.0/$h - $iw)/2)}]

        return ${pad}x0
    } else { 
        set pad [expr {round(($h*$iw*1.0/$w - $ih)/2)}]
        
        return 0x$pad
    } 
}


    
ad_proc -public phb::load_images {images base parent_id} { 
    set image {}
    set stat [catch { 
        foreach info $images { 
            unset image
            array set image $info
            
            set rel $image(relation_tag) 
            set image(name) ${base}_$image(relation_tag)
            set image(title) $image(name)
        
            # There should only ever be one and if exists more than one things have gone seriously wrong...
            set item_id [db_string image "select child_id from cr_child_rels where parent_id = :parent_id and relation_tag = :rel" -default {}]
            
            if {[empty_string_p $item_id]} { 
                set item_id [db_nextval acs_object_id_seq]            
                set image(new_image) 1
            } else {
                set image(new_image) 0
            } 
            set revision_id [db_nextval acs_object_id_seq]            
            set image(item_id) $item_id
            set image(revision_id) $revision_id
            set image(parent_id) $parent_id
            set image(is_live) "t"
            set image(filename) [cr_create_content_file $item_id $revision_id $image(file)]
            
            phb::insert_image image
        } 
    } errMsg]
    foreach info $images { 
        array set image $info
        catch {file delete -force -- $image(file)} 
    } 
    if { $stat } { 
        return -code $stat -errorinfo $errMsg
    } else { 
        return 
    } 
}

ad_proc -public phb::insert_image {image_ref} { 

    upvar $image_ref image 
    
    set params [list "creation_date   => sysdate" \
                    "creation_user   => [ad_conn user_id]" \
                    "creation_ip     => '[ad_conn peeraddr]'" ]
    if {$image(new_image)} { 
        # new image.
        foreach var { 
            name
            parent_id
            item_id
            revision_id
            context_id
            title
            description
            mime_type
            nls_language
            relation_tag
            is_live
            filename
            height
            width
            file_size 
        } { 
            if { [ info exists image($var) ] } { 
                set $var $image($var)
                lappend params "$var => :$var"
            }
        }
            
        set item_id [db_exec_plsql insert_image "
        begin
          :1 := image.new( [join $params ", "] );
        end;"]
    } else { 
        # New revision 
        foreach var { 
            item_id
            revision_id
            title
            description
            mime_type
            nls_language
            is_live
            filename
            height
            width
            file_size 
        } { 
            if { [ info exists image($var) ] } { 
                set $var $image($var)
                lappend params "$var => :$var"
            }
        }
            
        set item_id [db_exec_plsql insert_image "
        begin
          :1 := image.new_revision( [join $params ", "] );
        end;"]
    }
}

ad_proc -private phb::privacy_label {column relation name} {
    if {![string match "priv*" $column]} {
        return {}
    } 
        
    # special overrides. empty string means omit asking.
    array set special { 
        address.priv {home address}
        home.priv {home phone}
        mobile.priv {mobile phone}
        hometown.priv_address {}
        parent.priv_personal {}
        parent.priv_education {}
        parent.priv_employment {}
        parent.priv_address {}
        priv_gender {}
        priv_ethnicity {}
        priv_birthdate {}
        child children
        priv_hometown_city {}
        priv_hometown_state {}
        priv_hometown_country {}
        priv_hometown_postcode {}
        priv_hometown {}
        priv_address {home street address}
        priv_one_word {one word description}
        job_current.priv {current job}
        job_past.priv {past job}
    } 

    if {[info exists special($relation.$column)] } { 
        return $special($relation.$column)
    } elseif {[info exists special($column)] } { 
        return $special($column)
    } 


    regsub {^priv_?} $column {} label
    regsub {_} $label { } label
    if {[empty_string_p $label]} { 
        if {![string equal $relation personal_data] } { 
            return [lrange [split $name _] 2 end]
        }            
    } 
    return $label
}

ad_proc -private phb::privacy_widget {field label value} { 
    generate the ad_form stanza for a privacy widget
} { 
    if {[empty_string_p $label]} { 
        return {}
    } 
        
    return [list "$field:integer(checkbox)" \
                [list label $label] \
                {options {{Suppress 10}}} \
                [list value $value] \
               ]
}


ad_proc -private phb::join_array {
    ar pre
} { 
    given an array ref ar and prefix pre 
    generate a comma seperated list 
    of the prefixed fields.
} { 
    set j {}
    set out {}
    upvar $ar arr

    foreach i {1 2 3 4 5 6 7 8 9} { 
        if {[info exists arr(${pre}_$i)]
            && ![empty_string_p $arr(${pre}_$i)] } { 
            append out $j$arr(${pre}_$i)
            set j ", "
        } 
    } 
    return $out
} 

ad_proc -public phb::full_number {
    country_code area_code phone_number extension
} { 
    generate a formatted version of the full phone number 
} { 
    set full_number [list]
    if {![empty_string_p $phone_number]} {
        regsub { -+} $phone_number {} phone_number
        set cut [expr [string length $phone_number]/2]
        if {$cut > 1} { 
            set phone_number "[string range $phone_number 0 [expr $cut - 1]]-[string range $phone_number $cut end]"
        } 
        if {![empty_string_p $country_code]} { 
            regsub -all {[-+ ]} $country_code {} country_code
        } 
        if {[lsearch {{} +1 1 +} $country_code] == -1} { 
            lappend full_number "+$country_code"
        } 
        if {! [empty_string_p $area_code] } { 
            lappend full_number $area_code $phone_number
        } else { 
            lappend full_number "?-$phone_number"
        } 
    } 
    set full_number [join $full_number "-"]
    regsub {^\s*x} $extension {} extension
    if {![empty_string_p $extension]} { 
        append full_number " x$extension"
    }
    
    return $full_number
} 

ad_proc -public phb::span_dates { started ended } {
    generates a formatted version of a span date.
} { 

    set start_month [clock format [clock scan $started] -format "%B %Y"]
    set end_month [clock format [clock scan $ended] -format "%B %Y"]

    if {[string match "*-01-01" $started] 
        && [string match "*-01-01" $ended] } { 
        set started [lindex [split $started -] 0]
        set ended [lindex [split $ended -] 0]
        set dates "$started - $ended"
    } elseif {[empty_string_p $started]} { 
        if { [string match "*-01-01" $ended] } { 
            set ended [lindex [split $ended -] 0]
        } 
        set dates $ended
    } elseif {[empty_string_p $ended]} {
        if { [string match "*-01-01" $started] } { 
            set started [lindex [split $started -] 0]
        } 
        set dates $started
    } else {
        # both present
        if {[string match "*-01" $started] 
            && [string match "*-01" $ended] } { 
            set dates "$start_month - $end_month"
        } else { 
            set dates "$started - $ended" 
        } 
    } 
    
    return $dates
}

ad_proc -private phb::allowed_children_no_cache { 
    type 
} { 
    return a list of valid child type relation
} { 
    return [db_list_of_lists allowed {
        SELECT c.child_type, c.relation_tag
          FROM cr_type_children c
         WHERE c.parent_type = :type 
    } ]
}            

ad_proc -private phb::allowed_children { 
    type 
} { 
    return [util_memoize [list phb::allowed_children_no_cache $type]]
}            
    

ad_proc -public phb::users_items {
    -user_ids
} { 
    given the user_id returns the phb_person item id + 
    all children ids as returned by phb::related_items
    
    @param user_ids list user_ids to look up data for.

    @return array get format $type.$relation [list itemid ... ]

    @see phb::related_items
} { 
    set parent_id [phb::root_folder]

    set keys [list]

    foreach user_id $user_ids { 
        lappend keys '[phb::key [lindex $user_id 0]]'
    } 

    set key_count [llength $keys]

    set max_in_items 1000
    set root_item_ids [list]

    for {set i 0} {$key_count > [expr {$i * $max_in_items}]} {incr i} { 
        db_foreach item_lookup "select item_id from cr_items where name in ([join [lrange $keys [expr $i * $max_in_items] [expr {($i+1)*$max_in_items - 1}]] ","]) and parent_id = :parent_id" {
            lappend root_item_ids $item_id
        }
    }

    return [phb::related_items -parent_type phb_person -item_ids $root_item_ids]
} 

ad_proc -public phb::related_items { 
    -parent_type
    {-item_ids {}}
} { 
    Retrieves the related items for the given 
    item_id and returns them in array get form.
    if there are multiple children the array value 
    for the relation will contain a list.

    It will return an array element for every allowed relation for the content type
    plus the parent item_id as ($parent_type.parent $item_ids)

    @param item_id integer item id.  If the item id is not found the 
                   array will simply be a list of 
    
    @param parent_type the parent type for which to return the 

    @return child item ids for the item.  

    @see phb::users_items
} {
    # We create an entry for every allowed type
    foreach rel [phb::allowed_children $parent_type] { 
        set child([join $rel .]) {} 
    } 

    set child($parent_type.parent) $item_ids
    set id_csv [join $item_ids {, }]

    if {![empty_string_p $id_csv]} { 
        db_foreach children "SELECT child_type, c.relation_tag, rel.child_id
           FROM cr_type_children c, acs_object_types o, cr_child_rels rel
          WHERE c.child_type = o.object_type
            and c.parent_type = :parent_type 
            and rel.relation_tag = c.relation_tag
            and rel.parent_id in ($id_csv)" {
                lappend child($child_type.$relation_tag) $child_id
            }
    } 

    return [array get child]
}



ad_proc -private phb::validate::phone { 
    -prefix 
    -form
} { 
    validate that a phone entry looks ok.
} { 
    foreach var {country_code area_code phone_number} { 
        set $var [template::element::get_value $form $prefix$var]
    } 

    set country_code [string trim $country_code {+- }]

    if {[string equal 1 $country_code]
        || [empty_string_p $country_code]} { 
        set mash [string trim [string map {{ } {} - {} + {}} $phone_number]]
        if {![string is space $mash] 
            && ![regexp {^[1-9][0-9]{6}$} $mash]} { 
            template::element::set_error $form ${prefix}phone_number "Phone number must be 7 digits (or provide a country code for international numbers)"
        } 
        if {! [string is space $area_code]
            && ![regexp {^[1-9][0-9]{2}$} [string trim $area_code]]} { 
            template::element::set_error $form ${prefix}area_code "Area code must be 3 digits (or provide a country code for international numbers)"
        } 
    } else { 
        if {![string is digit $country_code]} { 
            template::element::set_error $form ${prefix}country_code "\"$country_code\" is not a numeric dialing code." 
        } elseif {![regexp {^[1-9][0-9]{0,2}$} $country_code]} {
            template::element::set_error $form ${prefix}country_code "\"$country_code\" is not a valid country code."
        } 
        
        if {![regexp {^[- 0-9]*$} $phone_number]} { 
            template::element::set_error $form ${prefix}phone_number "\"$phone_number\" invalid.  Only digits and - allowed."
        } 

        if {![regexp {^[- 0-9]*$} $area_code]} { 
            template::element::set_error $form ${prefix}area_code "\"$area_code\" invalid. Only digits and - allowed."
        }
    } 
} 

ad_proc -private phb::validate::address { 
    -prefix 
    -form
} { 
    validate that an address is ok.
} { 
    foreach var {postcode state country} { 
        set $var [template::element::get_value $form $prefix$var]
    } 

    set state [string toupper [string trim $state]]
    set postcode [string trim $postcode]

    if {[string equal $country US]} { 
        if {![empty_string_p $state] 
            && ![phb::validate::state_p $state] } { 
            template::element::set_error $form ${prefix}state "\"$state\" is not a valid two character state abbreviation."
        } else { 
            # if us and valid stuff in the upper version.
            template::element set_value $form ${prefix}state $state
        } 
        if {![empty_string_p $postcode] 
            && ![regexp {^[0-9]{5}(-[0-9]{4})?$} $postcode]} { 
            template::element::set_error $form ${prefix}postcode "\"$postcode\" is not a valid ZIP or ZIP+4 zipcode."
        } 
    } 
}
        
ad_proc -private phb::validate::state_p { 
    state
} { 
    check if a state abbrev is valid
} { 
    if {[lsearch \
              { 
                  AL AK AZ AR CA CO CT DE DC FL GA HI ID IL IN IA KS KY LA ME MD MA MI
                  MN MS MO MT NE NV NH NJ NM NY NC ND OH OK OR PA RI SC SD TN TX UT VT
                  VA WA WV WI WY AS GU MP PR VI FM UM 67 MH PW 71 76 79 81 84 86 89 95
              } [string toupper $state]] == -1 
    } { 
        return 0
    } else { 
        return 1
    } 
}


ad_proc -private phb::preinsert { form } { 
    Do some preinsert massaging of data 
} { 
    foreach prefix [list phnm_ phhm_] { 
        foreach var [list area_code country_code phone_number] { 
            if {![catch {set val [template::element get_value $form $prefix$var]}]} { 
                set val [string trim [string map {+ {} - {} { } {}} $val]]
                template::element set_properties $form $prefix$var value $val
            } 
        }
    }

    # trim names
    foreach prefix [list prsn_] {
        foreach var [list first_name middle_name last_name preferred_name former_name] { 
            if {![catch {set val [template::element get_value $form $prefix$var]}]} { 
                set val [string trim $val]
                template::element set_properties $form $prefix$var value $val
            } 
        }
    }
}

ad_proc -private phb::user_can_create_efl {
    user_id
} { 
    checks if the user is a dotlrn_member_rel relation to 
    a group with permissions to create an efl.
} { 
    set package_id [apm_package_id_from_key photobook] 

    return [db_string can_create_efl { 
        select count(*) 
        from dual 
        where exists ( 
                      select 1 
                      from acs_permissions p, acs_rels r 
                      where p.privilege = 'sloan_create_efl'
                        and p.object_id = :package_id
                        and p.grantee_id = r.object_id_one 
                        and r.object_id_two = :user_id
                        and r.rel_type = 'dotlrn_member_rel')
    } -default 0]
}

ad_proc -private phb::diff_pair {a b} { 
    generate the shortest string that is different between a and b
} {
    for {set i 0} {$i < 5} {incr i} { 
        if {![string equal [string range $a 0 $i] [string range $b 0 $i]]} {

            return [string range $b 0 $i]

        } 
    } 
    return $b
} 

          
ad_proc -private phb::pagination_label {users size page} { 
    generate pagination label 
} {
    if {$page < 2} { 
        set start {A}
    } else {
        set start [phb::diff_pair \
                       [lindex [lindex $users [expr ($page - 1) * $size - 1]] 2] \
                       [lindex [lindex $users [expr ($page - 1) * $size]] 2] ]
    } 

    if {[expr $page * $size] >= [llength $users]} { 
        set end {Z}
    } else { 
        set end [phb::diff_pair \
                     [lindex [lindex $users [expr $page * $size]] 2] \
                     [lindex [lindex $users [expr $page * $size - 1]] 2] ]
    } 
    
    return "$start - $end"
} 
        
ad_proc -public phb::pagination {name users size page} {
    generates a multirow for pagination 
    
    @parameter name  the name for the multirow datasource
    @parameter users list of users with elements {user_id role lastname}
    @parameter size  the nubmer of users per page
    @parameter page  the current page

    @return a multirow in the default level with elements page cur_p text
} { 
    set total [llength $users]
        
    template::multirow create $name page cur_p text 
    if { $total > $size} { 
        for {set i 1} {$i <= [expr {(($total - 1)/$size) + 1}]} {incr i} { 
            template::multirow append $name $i [expr {$i == $page}] [phb::pagination_label $users $size $i]
        } 
    }
}
